<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Unit\Plugin\views\filter;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\crm\CrmRelationshipTypeInterface;
use Drupal\crm\Plugin\views\filter\RelationshipStatisticsType;
use Drupal\Tests\UnitTestCase;

/**
 * Tests the RelationshipStatisticsType views filter plugin.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Plugin\views\filter\RelationshipStatisticsType
 */
class RelationshipStatisticsTypeTest extends UnitTestCase {

  /**
   * The filter plugin under test.
   *
   * @var \Drupal\crm\Plugin\views\filter\RelationshipStatisticsType
   */
  protected RelationshipStatisticsType $filterPlugin;

  /**
   * Mock entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Set up the container with string translation.
    $container = new ContainerBuilder();
    $translation = $this->createMock(TranslationInterface::class);
    $translation->method('translateString')
      ->willReturnCallback(function ($string) {
        return $string->getUntranslatedString();
      });
    $container->set('string_translation', $translation);

    // Create mock relationship types.
    $friends_type = $this->createMock(CrmRelationshipTypeInterface::class);
    $friends_type->method('id')->willReturn('friends');
    $friends_type->method('label')->willReturn('Friends');
    $friends_type->method('get')->willReturnCallback(function ($key) {
      return match ($key) {
        'label_a' => 'Friend',
        'label_b' => 'Friend',
        'asymmetric' => FALSE,
        default => NULL,
      };
    });

    $parent_child_type = $this->createMock(CrmRelationshipTypeInterface::class);
    $parent_child_type->method('id')->willReturn('parent_child');
    $parent_child_type->method('label')->willReturn('Parent-Child');
    $parent_child_type->method('get')->willReturnCallback(function ($key) {
      return match ($key) {
        'label_a' => 'Parent',
        'label_b' => 'Child',
        'asymmetric' => TRUE,
        default => NULL,
      };
    });

    $relationshipTypes = [
      'friends' => $friends_type,
      'parent_child' => $parent_child_type,
    ];

    // Create mock entity storage.
    $storage = $this->createMock(EntityStorageInterface::class);
    $storage->method('loadMultiple')
      ->willReturn($relationshipTypes);

    // Create mock entity type manager.
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->entityTypeManager->method('getStorage')
      ->with('crm_relationship_type')
      ->willReturn($storage);

    $container->set('entity_type.manager', $this->entityTypeManager);
    \Drupal::setContainer($container);

    // Create the filter plugin with minimal configuration.
    $configuration = [
      'field' => 'relationship_statistics_value',
      'table' => 'crm_contact__relationship_statistics',
    ];

    $this->filterPlugin = RelationshipStatisticsType::create(
      $container,
      $configuration,
      'crm_relationship_statistics_type',
      ['id' => 'crm_relationship_statistics_type']
    );
  }

  /**
   * Tests getValueOptions returns correct options for symmetric types.
   *
   * @covers ::getValueOptions
   */
  public function testGetValueOptionsSymmetric(): void {
    $options = $this->filterPlugin->getValueOptions();

    // Symmetric type should have one option with just the type ID.
    $this->assertArrayHasKey('friends', $options);
    $this->assertEquals('Friends', $options['friends']);
  }

  /**
   * Tests getValueOptions returns correct options for asymmetric types.
   *
   * @covers ::getValueOptions
   */
  public function testGetValueOptionsAsymmetric(): void {
    $options = $this->filterPlugin->getValueOptions();

    // Asymmetric type should have two options with positions.
    $this->assertArrayHasKey('parent_child:a', $options);
    $this->assertArrayHasKey('parent_child:b', $options);

    // Check that labels contain the position labels.
    $this->assertStringContainsString('Parent', (string) $options['parent_child:a']);
    $this->assertStringContainsString('Child', (string) $options['parent_child:b']);
  }

  /**
   * Tests getValueOptions returns all expected options.
   *
   * @covers ::getValueOptions
   */
  public function testGetValueOptionsCount(): void {
    $options = $this->filterPlugin->getValueOptions();

    // Should have 3 options: 1 symmetric + 2 asymmetric positions.
    $this->assertCount(3, $options);
  }

  /**
   * Tests getValueOptions caches results.
   *
   * @covers ::getValueOptions
   */
  public function testGetValueOptionsCaching(): void {
    $options1 = $this->filterPlugin->getValueOptions();
    $options2 = $this->filterPlugin->getValueOptions();

    // Should return the same array instance.
    $this->assertSame($options1, $options2);
  }

}
