<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Unit\Plugin\views\sort;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\TranslationInterface;
use Drupal\crm\CrmRelationshipTypeInterface;
use Drupal\crm\Plugin\views\sort\RelationshipStatisticsCount;
use Drupal\Tests\UnitTestCase;

/**
 * Tests the RelationshipStatisticsCount views sort plugin.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Plugin\views\sort\RelationshipStatisticsCount
 */
class RelationshipStatisticsCountTest extends UnitTestCase {

  /**
   * The sort plugin under test.
   *
   * @var \Drupal\crm\Plugin\views\sort\RelationshipStatisticsCount
   */
  protected RelationshipStatisticsCount $sortPlugin;

  /**
   * Mock entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Set up the container with string translation.
    $container = new ContainerBuilder();
    $translation = $this->createMock(TranslationInterface::class);
    $translation->method('translateString')
      ->willReturnCallback(function ($string) {
        return $string->getUntranslatedString();
      });
    $container->set('string_translation', $translation);

    // Create mock relationship types.
    $friends_type = $this->createMock(CrmRelationshipTypeInterface::class);
    $friends_type->method('id')->willReturn('friends');
    $friends_type->method('label')->willReturn('Friends');
    $friends_type->method('get')->willReturnCallback(function ($key) {
      return match ($key) {
        'label_a' => 'Friend',
        'label_b' => 'Friend',
        'asymmetric' => FALSE,
        default => NULL,
      };
    });

    $parent_child_type = $this->createMock(CrmRelationshipTypeInterface::class);
    $parent_child_type->method('id')->willReturn('parent_child');
    $parent_child_type->method('label')->willReturn('Parent-Child');
    $parent_child_type->method('get')->willReturnCallback(function ($key) {
      return match ($key) {
        'label_a' => 'Parent',
        'label_b' => 'Child',
        'asymmetric' => TRUE,
        default => NULL,
      };
    });

    $relationshipTypes = [
      'friends' => $friends_type,
      'parent_child' => $parent_child_type,
    ];

    // Create mock entity storage.
    $storage = $this->createMock(EntityStorageInterface::class);
    $storage->method('loadMultiple')
      ->willReturn($relationshipTypes);

    // Create mock entity type manager.
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->entityTypeManager->method('getStorage')
      ->with('crm_relationship_type')
      ->willReturn($storage);

    $container->set('entity_type.manager', $this->entityTypeManager);
    \Drupal::setContainer($container);

    // Create the sort plugin with minimal configuration.
    $configuration = [
      'field' => 'relationship_statistics_count',
      'table' => 'crm_contact__relationship_statistics',
    ];

    $this->sortPlugin = RelationshipStatisticsCount::create(
      $container,
      $configuration,
      'crm_relationship_statistics_count',
      ['id' => 'crm_relationship_statistics_count']
    );
  }

  /**
   * Tests defineOptions returns expected defaults.
   *
   * @covers ::defineOptions
   */
  public function testDefineOptions(): void {
    $options = $this->invokeMethod($this->sortPlugin, 'defineOptions');

    $this->assertArrayHasKey('relationship_type', $options);
    $this->assertEquals('', $options['relationship_type']['default']);
    $this->assertArrayHasKey('order', $options);
  }

  /**
   * Tests getRelationshipTypeOptions correct options for symmetric types.
   *
   * @covers ::getRelationshipTypeOptions
   */
  public function testGetRelationshipTypeOptionsSymmetric(): void {
    $options = $this->invokeMethod($this->sortPlugin, 'getRelationshipTypeOptions');

    // Symmetric type should have one option with just the type ID.
    $this->assertArrayHasKey('friends', $options);
    $this->assertEquals('Friends', $options['friends']);
  }

  /**
   * Tests getRelationshipTypeOptions correct options for asymmetric types.
   *
   * @covers ::getRelationshipTypeOptions
   */
  public function testGetRelationshipTypeOptionsAsymmetric(): void {
    $options = $this->invokeMethod($this->sortPlugin, 'getRelationshipTypeOptions');

    // Asymmetric type should have two options with positions.
    $this->assertArrayHasKey('parent_child:a', $options);
    $this->assertArrayHasKey('parent_child:b', $options);

    // Check that labels contain the position labels.
    $this->assertStringContainsString('Parent', (string) $options['parent_child:a']);
    $this->assertStringContainsString('Child', (string) $options['parent_child:b']);
  }

  /**
   * Tests getRelationshipTypeOptions returns all expected options.
   *
   * @covers ::getRelationshipTypeOptions
   */
  public function testGetRelationshipTypeOptionsCount(): void {
    $options = $this->invokeMethod($this->sortPlugin, 'getRelationshipTypeOptions');

    // Should have 3 options: 1 symmetric + 2 asymmetric positions.
    $this->assertCount(3, $options);
  }

  /**
   * Invokes a protected/private method on an object.
   *
   * @param object $object
   *   The object to invoke the method on.
   * @param string $methodName
   *   The method name to invoke.
   * @param array $parameters
   *   Parameters to pass to the method.
   *
   * @return mixed
   *   The result of the method call.
   */
  protected function invokeMethod($object, string $methodName, array $parameters = []) {
    $reflection = new \ReflectionClass(get_class($object));
    $method = $reflection->getMethod($methodName);
    $method->setAccessible(TRUE);
    return $method->invokeArgs($object, $parameters);
  }

}
