<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Unit;

use Drupal\Core\Utility\ThemeRegistry;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityDisplayRepositoryInterface;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Theme\Registry;
use Drupal\crm\UserContactMappingViewBuilder;
use Drupal\Tests\UnitTestCase;

/**
 * Tests UserContactMappingViewBuilder functionality.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\UserContactMappingViewBuilder
 */
class UserContactMappingViewBuilderTest extends UnitTestCase {

  /**
   * The view builder under test.
   *
   * @var \Drupal\crm\UserContactMappingViewBuilder
   */
  protected $viewBuilder;

  /**
   * Mock entity type.
   *
   * @var \Drupal\Core\Entity\EntityTypeInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityType;

  /**
   * Mock entity repository.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityRepository;

  /**
   * Mock language manager.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $languageManager;

  /**
   * Mock theme registry.
   *
   * @var \Drupal\Core\Theme\Registry|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $themeRegistry;

  /**
   * Mock entity display repository.
   *
   * @var \Drupal\Core\Entity\EntityDisplayRepositoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityDisplayRepository;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Set up the container with required services.
    $container = new ContainerBuilder();
    $moduleHandler = $this->createMock(ModuleHandlerInterface::class);
    $moduleHandler->method('invokeAll')->willReturn([]);
    $moduleHandler->method('alter')->willReturn(NULL);
    $container->set('module_handler', $moduleHandler);
    \Drupal::setContainer($container);

    $this->entityType = $this->createMock(EntityTypeInterface::class);
    $this->entityType->method('id')->willReturn('crm_user_contact_mapping');

    $this->entityRepository = $this->createMock(EntityRepositoryInterface::class);
    $this->languageManager = $this->createMock(LanguageManagerInterface::class);

    // Mock theme registry with runtime that returns FALSE for template check.
    // getRuntime() returns a ThemeRegistry from Drupal\Core\Utility namespace.
    $themeRegistryRuntime = $this->createMock(ThemeRegistry::class);
    $themeRegistryRuntime->method('has')->willReturn(FALSE);
    $this->themeRegistry = $this->createMock(Registry::class);
    $this->themeRegistry->method('getRuntime')->willReturn($themeRegistryRuntime);

    $this->entityDisplayRepository = $this->createMock(EntityDisplayRepositoryInterface::class);

    $this->viewBuilder = new UserContactMappingViewBuilder(
      $this->entityType,
      $this->entityRepository,
      $this->languageManager,
      $this->themeRegistry,
      $this->entityDisplayRepository
    );
  }

  /**
   * Creates a mock entity with all required methods for getBuildDefaults.
   *
   * @return \Drupal\Core\Entity\EntityInterface|\PHPUnit\Framework\MockObject\MockObject
   *   The mock entity.
   */
  protected function createMockEntityForViewBuilder() {
    $mockEntity = $this->createMock(EntityInterface::class);
    $mockEntity->method('getCacheContexts')->willReturn([]);
    $mockEntity->method('getCacheTags')->willReturn(['crm_user_contact_mapping:1']);
    $mockEntity->method('getCacheMaxAge')->willReturn(-1);
    // Additional methods needed by EntityViewBuilder::getBuildDefaults().
    $mockEntity->method('isNew')->willReturn(TRUE);
    return $mockEntity;
  }

  /**
   * Tests that getBuildDefaults removes the #theme key.
   *
   * @covers ::getBuildDefaults
   */
  public function testGetBuildDefaultsRemovesThemeKey(): void {
    $mockEntity = $this->createMockEntityForViewBuilder();

    // Use reflection to call the protected method.
    $reflectionMethod = new \ReflectionMethod(UserContactMappingViewBuilder::class, 'getBuildDefaults');

    $result = $reflectionMethod->invoke($this->viewBuilder, $mockEntity, 'full');

    // Verify that #theme is not set.
    $this->assertArrayNotHasKey('#theme', $result);
  }

  /**
   * Tests that getBuildDefaults still returns expected build array keys.
   *
   * @covers ::getBuildDefaults
   */
  public function testGetBuildDefaultsReturnsExpectedKeys(): void {
    $mockEntity = $this->createMockEntityForViewBuilder();

    $reflectionMethod = new \ReflectionMethod(UserContactMappingViewBuilder::class, 'getBuildDefaults');

    $result = $reflectionMethod->invoke($this->viewBuilder, $mockEntity, 'full');

    // Verify expected keys are present from parent.
    $this->assertArrayHasKey('#view_mode', $result);
    $this->assertEquals('full', $result['#view_mode']);
    $this->assertArrayHasKey('#cache', $result);
  }

  /**
   * Tests getBuildDefaults with different view modes.
   *
   * @covers ::getBuildDefaults
   * @dataProvider viewModeProvider
   */
  public function testGetBuildDefaultsWithDifferentViewModes(string $viewMode): void {
    $mockEntity = $this->createMockEntityForViewBuilder();

    $reflectionMethod = new \ReflectionMethod(UserContactMappingViewBuilder::class, 'getBuildDefaults');

    $result = $reflectionMethod->invoke($this->viewBuilder, $mockEntity, $viewMode);

    // Theme should be unset regardless of view mode.
    $this->assertArrayNotHasKey('#theme', $result);
    $this->assertEquals($viewMode, $result['#view_mode']);
  }

  /**
   * Data provider for view modes.
   *
   * @return array
   *   View mode test data.
   */
  public static function viewModeProvider(): array {
    return [
      'full view mode' => ['full'],
      'teaser view mode' => ['teaser'],
      'default view mode' => ['default'],
    ];
  }

}
