<?php

declare(strict_types=1);

namespace Drupal\crm\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Render\RendererInterface;
use Drupal\crm\CrmContactInterface;
use Drupal\Core\Access\AccessResult;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Controller for the comment page.
 */
class CommentController extends ControllerBase {

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * Constructs a new CommentController.
   *
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   */
  public function __construct(RendererInterface $renderer) {
    $this->renderer = $renderer;
  }

  /**
   * Creates a new CommentController.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The container.
   *
   * @return static
   *   The created controller.
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('renderer')
    );
  }

  /**
   * Builds the comments page.
   *
   * @param \Drupal\crm\CrmContactInterface $crm_contact
   *   The contact.
   *
   * @return array
   *   A render array.
   */
  public function commentsPage(CrmContactInterface $crm_contact) {
    $contact_id = $crm_contact->id();
    $build = [
      '#cache' => [
        'tags' => [
          'crm_contact:' . $contact_id,
        ],
      ],
      '#title' => $this->t('Comments about @label', ['@label' => $crm_contact->label()]),
    ];

    // Assumes the comment field is 'comment'.
    $field_name = 'comment';

    // Show comment list.
    if ($crm_contact->hasField($field_name)) {
      $build['comment'] = $this->entityTypeManager()
        ->getViewBuilder('crm_contact')
        ->viewField($crm_contact->get($field_name), 'comment');

      $build['comment'][0]['comment_form']['#lazy_builder'] = [
        'crm.comment.lazy_builders:renderForm',
        [
          $crm_contact->getEntityTypeId(),
          $crm_contact->id(),
          $field_name,
          'crm_contact',
        ],
      ];
    }

    return $build;
  }

  /**
   * The title of the comments page.
   *
   * @param \Drupal\crm\CrmContactInterface $crm_contact
   *   The contact.
   *
   * @return \Drupal\Core\StringTranslation\TranslatableMarkup
   *   The title of the comments page.
   */
  public function title(CrmContactInterface $crm_contact) {
    return $this->t('Comments for @label', ['@label' => $crm_contact->label()]);
  }

  /**
   * Access check for the comments page.
   *
   * @param \Drupal\crm\CrmContactInterface $crm_contact
   *   The contact.
   *
   * @return \Drupal\Core\Access\AccessResult
   *   The access result.
   */
  public function access(CrmContactInterface $crm_contact) {
    return AccessResult::allowedIf($crm_contact->access('view') && $this->currentUser()->hasPermission('access comments'));
  }

}
