<?php

namespace Drupal\crm\Form;

use Drupal\Core\Entity\EntityForm;
use Drupal\Core\Form\FormStateInterface;

/**
 * Detail Type form.
 *
 * @property \Drupal\crm\CrmDetailTypeInterface $entity
 */
class DetailTypeForm extends EntityForm {

  const SAVED_NEW = 1;
  const SAVED_UPDATED = 2;

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {

    $form = parent::form($form, $form_state);

    $form['label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label'),
      '#maxlength' => 255,
      '#default_value' => $this->entity->label(),
      '#description' => $this->t('Label for the detail type.'),
      '#required' => TRUE,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $this->entity->id(),
      '#machine_name' => [
        'exists' => '\Drupal\crm\Entity\DetailType::load',
      ],
      '#disabled' => !$this->entity->isNew(),
    ];

    $form['status'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enabled'),
      '#default_value' => $this->entity->status(),
    ];

    $form['description'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Description'),
      '#default_value' => $this->entity->get('description'),
      '#description' => $this->t('Description of the detail type.'),
    ];

    $bundles = $this->entityTypeManager->getStorage('crm_contact_detail_type')->loadMultiple();
    $bundle_options = array_map(function ($type) {
      return $type->label();
    }, $bundles);

    $form['bundles'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Bundles'),
      '#options' => $bundle_options,
      '#default_value' => $this->entity->get('bundles') ?: [],
      '#description' => empty($bundle_options)
        ? $this->t('No contact detail types are available. Create contact detail types first.')
        : $this->t('Select the bundles that this detail type applies to.'),
      '#multiple' => TRUE,
    ];

    $form['negate'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Negate'),
      '#default_value' => $this->entity->get('negate') ?: FALSE,
      '#description' => $this->t('If checked, this detail type will be applied to all bundles except those selected.'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {

    $bundles = $form_state->getValue('bundles');
    $this->entity->set('bundles', array_filter(array_values($bundles)));

    $result = parent::save($form, $form_state);
    $message_args = ['%label' => $this->entity->label()];
    $message = $result == self::SAVED_NEW
      ? $this->t('Created new detail type %label.', $message_args)
      : $this->t('Updated detail type %label.', $message_args);
    $this->messenger()->addStatus($message);
    $form_state->setRedirectUrl($this->entity->toUrl('collection'));
    return $result;
  }

}
