<?php

declare(strict_types=1);

namespace Drupal\crm\Plugin\EntityReferenceSelection;

use Drupal\Core\Entity\Plugin\EntityReferenceSelection\DefaultSelection;

/**
 * Filters config entities based on the referencing entity's bundle.
 *
 * @EntityReferenceSelection(
 *   id = "default:crm_detail_type",
 *   label = @Translation("Default: CRM Detail Type"),
 *   entity_types = {"crm_detail_type"},
 *   group = "default",
 *   weight = 0
 * )
 */
class DetailTypeSelection extends DefaultSelection {

  /**
   * {@inheritdoc}
   */
  public function buildEntityQuery($match = NULL, $match_operator = 'CONTAINS') {

    $query = parent::buildEntityQuery($match, $match_operator);
    /** @var \Drupal\Core\Entity\ContentEntityInterface $referencing_entity */
    $referencing_entity = $this->configuration['entity'];
    $entity_type_id = $referencing_entity->getEntityTypeId();
    if ($entity_type_id !== 'crm_contact_detail') {
      return $query;
    }

    // This assumes the referencing entity is available during form building.
    $bundle = $referencing_entity->bundle();

    $query->condition('status', TRUE);

    // Case 1: Bundles is empty.
    $empty = $query->andConditionGroup()
      ->condition('bundles.0', NULL, 'IS NULL');

    // Case 2: negate = FALSE and bundle is in bundles.
    $match_bundle = $query->andConditionGroup()
      ->condition('negate', FALSE)
      ->condition('bundles.*', $bundle, 'CONTAINS');

    // Case 3: negate = TRUE and bundle is in bundles.
    // Entities excluded by bundle.
    $not_query = parent::buildEntityQuery($match, $match_operator);
    $not_query->condition('negate', TRUE)
      ->condition('bundles.*', $bundle, 'CONTAINS');

    $not_ids = $not_query->execute();

    // The entities that are negated and do not match the bundle.
    $ton_query = parent::buildEntityQuery($match, $match_operator);
    $ton_query->condition('negate', TRUE)
      ->condition('id', $not_ids, 'NOT IN');

    $not_bundle = $query->andConditionGroup()
      ->condition('id', $ton_query->execute(), 'IN');

    $or = $query->orConditionGroup()
      ->condition($empty)
      ->condition($not_bundle)
      ->condition($match_bundle);

    $query->condition($or);

    return $query;
  }

}
