<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel\Field;

use Drupal\crm\Entity\Relationship;
use Drupal\crm\Entity\RelationshipType;
use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\ContactType;
use Drupal\KernelTests\KernelTestBase;

/**
 * Tests the RelationshipContactsItemList computed field.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Field\RelationshipContactsItemList
 */
class RelationshipContactsItemListTest extends KernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'comment',
    'crm',
    'datetime',
    'field',
    'name',
    'system',
    'text',
    'user',
  ];

  /**
   * A relationship type entity.
   *
   * @var \Drupal\crm\Entity\RelationshipType
   */
  protected $relationshipType;

  /**
   * A contact type entity.
   *
   * @var \Drupal\crm\Entity\ContactType
   */
  protected $contactType;

  /**
   * Contact entities for testing.
   *
   * @var \Drupal\crm\Entity\Contact[]
   */
  protected $contacts;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_type');
    $this->installEntitySchema('crm_relationship');
    $this->installEntitySchema('crm_relationship_type');
    $this->installSchema('system', ['sequences']);

    // Create a contact type.
    $this->contactType = ContactType::create([
      'id' => 'person',
      'label' => 'Person',
      'date' => [
        'start_date' => [
          'label' => 'Date of birth',
        ],
        'end_date' => [
          'label' => 'Date of death',
        ],
      ],
    ]);
    $this->contactType->save();

    // Create a relationship type.
    $this->relationshipType = RelationshipType::create([
      'id' => 'test_relationship',
      'label' => 'Test Relationship',
    ]);
    $this->relationshipType->save();

    // Create test contacts.
    $this->contacts[0] = Contact::create([
      'bundle' => 'person',
      'name' => 'John Doe',
    ]);
    $this->contacts[0]->save();

    $this->contacts[1] = Contact::create([
      'bundle' => 'person',
      'name' => 'Jane Doe',
    ]);
    $this->contacts[1]->save();

    $this->contacts[2] = Contact::create([
      'bundle' => 'person',
      'name' => 'Bob Smith',
    ]);
    $this->contacts[2]->save();
  }

  /**
   * Tests computation of contact_a field value.
   *
   * @covers ::computeValue
   */
  public function testComputeValueContactA() {
    $relationship = Relationship::create([
      'bundle' => 'test_relationship',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
    ]);

    $contact_a_field = $relationship->get('contact_a');
    $contact_a_value = $contact_a_field->getValue();

    $this->assertEquals($this->contacts[0]->id(), $contact_a_value[0]['target_id']);
  }

  /**
   * Tests computation of contact_b field value.
   *
   * @covers ::computeValue
   */
  public function testComputeValueContactB() {
    $relationship = Relationship::create([
      'bundle' => 'test_relationship',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
    ]);

    $contact_b_field = $relationship->get('contact_b');
    $contact_b_value = $contact_b_field->getValue();

    $this->assertEquals($this->contacts[1]->id(), $contact_b_value[0]['target_id']);
  }

  /**
   * Tests computation with only one contact in the contacts field.
   *
   * @covers ::computeValue
   */
  public function testComputeValueWithSingleContact() {
    $relationship = Relationship::create([
      'bundle' => 'test_relationship',
      'contacts' => [$this->contacts[0]->id()],
    ]);

    // Contact A should work with single contact.
    $contact_a_field = $relationship->get('contact_a');
    $contact_a_value = $contact_a_field->getValue();
    $this->assertEquals($this->contacts[0]->id(), $contact_a_value[0]['target_id']);

    // Contact B should be null/empty when there's no second contact.
    $contact_b_field = $relationship->get('contact_b');
    $contact_b_value = $contact_b_field->getValue();
    $this->assertNull($contact_b_value[0]['target_id']);
  }

  /**
   * Tests computation with empty contacts field.
   *
   * @covers ::computeValue
   * @covers ::referencedEntities
   */
  public function testComputeValueWithEmptyContacts() {
    $relationship = Relationship::create([
      'bundle' => 'test_relationship',
      'contacts' => [],
    ]);

    // Both fields should handle empty contacts gracefully.
    $contact_a_field = $relationship->get('contact_a');
    $contact_a_value = $contact_a_field->getValue();
    $this->assertNull($contact_a_value[0]['target_id']);

    $contact_b_field = $relationship->get('contact_b');
    $contact_b_value = $contact_b_field->getValue();
    $this->assertNull($contact_b_value[0]['target_id']);

    // Test referencedEntities with empty contacts.
    $contact_a_entities = $contact_a_field->referencedEntities();
    $this->assertIsArray($contact_a_entities);
    $this->assertCount(0, $contact_a_entities);

    $contact_b_entities = $contact_b_field->referencedEntities();
    $this->assertIsArray($contact_b_entities);
    $this->assertCount(0, $contact_b_entities);
  }

  /**
   * Tests setValue method for contact_a field.
   *
   * @covers ::setValue
   */
  public function testSetValueContactA() {
    $relationship = Relationship::create([
      'bundle' => 'test_relationship',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
    ]);

    // Change contact_a to contacts[2].
    $contact_a_field = $relationship->get('contact_a');
    $contact_a_field->setValue([['target_id' => $this->contacts[2]->id()]]);

    // Verify the underlying contacts field was updated.
    $contacts_values = $relationship->get('contacts')->getValue();
    $this->assertEquals($this->contacts[2]->id(), $contacts_values[0]['target_id']);
    $this->assertEquals($this->contacts[1]->id(), $contacts_values[1]['target_id']);
  }

  /**
   * Tests setValue method for contact_b field.
   *
   * @covers ::setValue
   */
  public function testSetValueContactB() {
    $relationship = Relationship::create([
      'bundle' => 'test_relationship',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
    ]);

    // Change contact_b to contacts[2].
    $contact_b_field = $relationship->get('contact_b');
    $contact_b_field->setValue([['target_id' => $this->contacts[2]->id()]]);

    // Verify the underlying contacts field was updated.
    $contacts_values = $relationship->get('contacts')->getValue();
    $this->assertEquals($this->contacts[0]->id(), $contacts_values[0]['target_id']);
    $this->assertEquals($this->contacts[2]->id(), $contacts_values[1]['target_id']);
  }

  /**
   * Tests setValue with scalar value instead of array.
   *
   * @covers ::setValue
   */
  public function testSetValueWithScalar() {
    $relationship = Relationship::create([
      'bundle' => 'test_relationship',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
    ]);

    // Set contact_a with scalar value.
    $contact_a_field = $relationship->get('contact_a');
    $contact_a_field->setValue($this->contacts[2]->id());

    // Verify the underlying contacts field was updated.
    $contacts_values = $relationship->get('contacts')->getValue();
    $this->assertEquals($this->contacts[2]->id(), $contacts_values[0]['target_id']);
    $this->assertEquals($this->contacts[1]->id(), $contacts_values[1]['target_id']);
  }

  /**
   * Tests setValue on empty contacts field expands array.
   *
   * @covers ::setValue
   */
  public function testSetValueExpandsContactsArray() {
    $relationship = Relationship::create([
      'bundle' => 'test_relationship',
      'contacts' => [],
    ]);

    // Set contact_b (index 1) when contacts is empty.
    $contact_b_field = $relationship->get('contact_b');
    $contact_b_field->setValue([['target_id' => $this->contacts[1]->id()]]);

    // Verify the contacts array was expanded with null placeholders.
    $contacts_values = $relationship->get('contacts')->getValue();
    $this->assertCount(2, $contacts_values);
    $this->assertNull($contacts_values[0]['target_id']);
    $this->assertEquals($this->contacts[1]->id(), $contacts_values[1]['target_id']);
  }

  /**
   * Tests setValue with empty values.
   *
   * @covers ::setValue
   */
  public function testSetValueWithEmptyValues() {
    $relationship = Relationship::create([
      'bundle' => 'test_relationship',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
    ]);

    $original_contacts = $relationship->get('contacts')->getValue();

    // Set empty values should not change the underlying contacts field.
    $contact_a_field = $relationship->get('contact_a');
    $contact_a_field->setValue([]);

    $contacts_after = $relationship->get('contacts')->getValue();
    $this->assertEquals($original_contacts, $contacts_after);
  }

  /**
   * Tests referencedEntities method for contact_a field.
   *
   * @covers ::referencedEntities
   */
  public function testReferencedEntitiesContactA() {
    $relationship = Relationship::create([
      'bundle' => 'test_relationship',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
    ]);

    $contact_a_field = $relationship->get('contact_a');
    $referenced_entities = $contact_a_field->referencedEntities();

    $this->assertCount(1, $referenced_entities);
    $this->assertEquals($this->contacts[0]->id(), $referenced_entities[0]->id());
  }

  /**
   * Tests referencedEntities method for contact_b field.
   *
   * @covers ::referencedEntities
   */
  public function testReferencedEntitiesContactB() {
    $relationship = Relationship::create([
      'bundle' => 'test_relationship',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
    ]);

    $contact_b_field = $relationship->get('contact_b');
    $referenced_entities = $contact_b_field->referencedEntities();

    $this->assertCount(1, $referenced_entities);
    $this->assertEquals($this->contacts[1]->id(), $referenced_entities[0]->id());
  }

  /**
   * Tests referencedEntities with missing contact.
   *
   * @covers ::referencedEntities
   */
  public function testReferencedEntitiesWithMissingContact() {
    $relationship = Relationship::create([
      'bundle' => 'test_relationship',
      'contacts' => [$this->contacts[0]->id()],
    ]);

    // Contact A should return the first contact.
    $contact_a_field = $relationship->get('contact_a');
    $referenced_entities_a = $contact_a_field->referencedEntities();
    $this->assertCount(1, $referenced_entities_a);
    $this->assertEquals($this->contacts[0]->id(), $referenced_entities_a[0]->id());

    // Contact B should return empty array when there's no second contact.
    $contact_b_field = $relationship->get('contact_b');
    $referenced_entities_b = $contact_b_field->referencedEntities();
    $this->assertIsArray($referenced_entities_b);
    $this->assertCount(0, $referenced_entities_b);
  }

  /**
   * Tests field integration with relationship entity.
   *
   * @covers ::computeValue
   * @covers ::referencedEntities
   */
  public function testFieldIntegrationWithRelationship() {
    $relationship = Relationship::create([
      'bundle' => 'test_relationship',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
    ]);
    $relationship->save();

    // Test that computed fields work after entity save/load cycle.
    $loaded_relationship = Relationship::load($relationship->id());

    $contact_a_field = $loaded_relationship->get('contact_a');
    $contact_a_value = $contact_a_field->getValue();
    $this->assertEquals($this->contacts[0]->id(), $contact_a_value[0]['target_id']);

    $contact_b_field = $loaded_relationship->get('contact_b');
    $contact_b_value = $contact_b_field->getValue();
    $this->assertEquals($this->contacts[1]->id(), $contact_b_value[0]['target_id']);

    // Test referenced entities after save/load.
    $contact_a_entities = $contact_a_field->referencedEntities();
    $this->assertEquals($this->contacts[0]->id(), $contact_a_entities[0]->id());

    $contact_b_entities = $contact_b_field->referencedEntities();
    $this->assertEquals($this->contacts[1]->id(), $contact_b_entities[0]->id());
  }

  /**
   * Tests that the field implements the correct interface.
   */
  public function testFieldImplementsInterface() {
    $relationship = Relationship::create([
      'bundle' => 'test_relationship',
      'contacts' => [$this->contacts[0]->id(), $this->contacts[1]->id()],
    ]);

    $contact_a_field = $relationship->get('contact_a');
    $this->assertInstanceOf('\Drupal\Core\Field\EntityReferenceFieldItemListInterface', $contact_a_field);

    $contact_b_field = $relationship->get('contact_b');
    $this->assertInstanceOf('\Drupal\Core\Field\EntityReferenceFieldItemListInterface', $contact_b_field);
  }

}
