<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Unit\Form;

use Drupal\Tests\UnitTestCase;
use Drupal\crm\Form\ContactTypeForm;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\crm\Entity\ContactType;

/**
 * Contact type form test.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Form\ContactTypeForm
 * @uses \Drupal\crm\Form\ContactTypeForm
 */
class ContactTypeFormTest extends UnitTestCase {

  const SAVED_NEW = 1;
  const SAVED_UPDATED = 2;

  /**
   * The form under test.
   *
   * @var \Drupal\crm\Form\ContactTypeForm
   */
  protected $form;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $messenger;

  /**
   * The entity storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $storage;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The contact type entity.
   *
   * @var \Drupal\crm\Entity\ContactType|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entity;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $container = new ContainerBuilder();

    $string_translation = $this->getStringTranslationStub();
    $container->set('string_translation', $string_translation);

    $this->messenger = $this->createMock(MessengerInterface::class);
    $container->set('messenger', $this->messenger);

    $this->storage = $this->createMock(EntityStorageInterface::class);
    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->entityTypeManager->expects($this->any())
      ->method('getStorage')
      ->with('crm_relationship_type')
      ->willReturn($this->storage);

    $container->set('entity_type.manager', $this->entityTypeManager);

    \Drupal::setContainer($container);

    $this->form = ContactTypeForm::create($container);

    $this->entity = $this->createMock(ContactType::class);
    $this->form->setEntity($this->entity);
  }

  /**
   * Tests the create method.
   *
   * @covers ::create
   */
  public function testCreate(): void {
    $container = \Drupal::getContainer();
    $form = ContactTypeForm::create($container);

    $this->assertInstanceOf(ContactTypeForm::class, $form);
  }

  /**
   * Test the constructor.
   *
   * @covers ::__construct
   */
  public function testConstructor(): void {
    $storage = $this->createMock(EntityStorageInterface::class);
    $form = new ContactTypeForm($storage);
    $this->assertInstanceOf(ContactTypeForm::class, $form);
  }

  /**
   * Test the form id.
   *
   * @covers ::getFormId
   */
  public function testGetFormId(): void {
    $entity_type = $this->createMock(EntityTypeInterface::class);
    $entity_type->expects($this->once())
      ->method('hasKey')
      ->with('bundle')
      ->willReturn(FALSE);

    $this->entity->expects($this->once())
      ->method('getEntityTypeId')
      ->willReturn('crm_contact_type');
    $this->entity->expects($this->once())
      ->method('getEntityType')
      ->willReturn($entity_type);

    $this->form->setOperation('edit');

    $this->assertEquals('crm_contact_type_edit_form', $this->form->getFormId());
  }

  /**
   * Test the form build.
   *
   * @covers ::form
   */
  public function testForm(): void {
    $entity_type = $this->createMock(EntityTypeInterface::class);
    $entity_type->expects($this->once())
      ->method('getKey')
      ->with('id')
      ->willReturn('id');

    $this->entity->expects($this->once())
      ->method('getEntityType')
      ->willReturn($entity_type);
    $this->entity->expects($this->any())
      ->method('label')
      ->willReturn('Test Contact Type');
    $this->entity->expects($this->any())
      ->method('id')
      ->willReturn('test_contact_type');
    $this->entity->expects($this->any())
      ->method('get')
      ->willReturnMap([
        ['description', 'Test description'],
        ['date', [
          'start_date' => ['label' => 'Start Date', 'description' => 'Start description'],
          'end_date' => ['label' => 'End Date', 'description' => 'End description'],
        ],
        ],
      ]);

    $form_state = $this->createMock(FormStateInterface::class);
    $form = $this->form->form([], $form_state);

    $this->assertIsArray($form);
    $this->assertArrayHasKey('label', $form);
    $this->assertArrayHasKey('id', $form);
    $this->assertArrayHasKey('description', $form);
    $this->assertArrayHasKey('additional_settings', $form);
    $this->assertArrayHasKey('start_date', $form);
    $this->assertArrayHasKey('end_date', $form);

    // Test label field.
    $this->assertEquals('textfield', $form['label']['#type']);
    $this->assertEquals('Test Contact Type', $form['label']['#default_value']);
    $this->assertTrue($form['label']['#required']);

    // Test id field (machine name)
    $this->assertEquals('machine_name', $form['id']['#type']);
    $this->assertEquals('test_contact_type', $form['id']['#default_value']);

    // Test description field.
    $this->assertEquals('textarea', $form['description']['#type']);
    $this->assertEquals('Test description', $form['description']['#default_value']);

    // Test vertical tabs.
    $this->assertEquals('vertical_tabs', $form['additional_settings']['#type']);

    // Test start date section.
    $this->assertEquals('details', $form['start_date']['#type']);
    $this->assertEquals('additional_settings', $form['start_date']['#group']);
    $this->assertArrayHasKey('start_date_label', $form['start_date']);
    $this->assertArrayHasKey('start_date_description', $form['start_date']);

    // Test end date section.
    $this->assertEquals('details', $form['end_date']['#type']);
    $this->assertEquals('additional_settings', $form['end_date']['#group']);
    $this->assertArrayHasKey('end_date_label', $form['end_date']);
    $this->assertArrayHasKey('end_date_description', $form['end_date']);
  }

  /**
   * Test the form build with edit operation.
   *
   * @covers ::form
   */
  public function testFormWithEditOperation(): void {
    $entity_type = $this->createMock(EntityTypeInterface::class);
    $entity_type->expects($this->once())
      ->method('getKey')
      ->with('id')
      ->willReturn('id');

    $this->entity->expects($this->once())
      ->method('getEntityType')
      ->willReturn($entity_type);
    $this->entity->expects($this->any())
      ->method('label')
      ->willReturn('Test Contact Type');
    $this->entity->expects($this->any())
      ->method('id')
      ->willReturn('test_contact_type');
    $this->entity->expects($this->any())
      ->method('get')
      ->willReturnMap([
        ['description', 'Test description'],
        ['date', []],
      ]);

    $this->form->setOperation('edit');

    $form_state = $this->createMock(FormStateInterface::class);
    $form = $this->form->form([], $form_state);

    $this->assertArrayHasKey('#title', $form);
    $expected_title = $this->getStringTranslationStub()->translate('Edit %label contact type', ['%label' => 'Test Contact Type']);
    $this->assertEquals($expected_title, $form['#title']);
  }

  /**
   * Test the actions method.
   *
   * @covers ::actions
   */
  public function testActions(): void {
    $form_state = $this->createMock(FormStateInterface::class);

    // We need to set up the entity and form properly to call actions.
    $this->entity->expects($this->any())
      ->method('isNew')
      ->willReturn(FALSE);

    // Use Reflection to access protected method.
    $reflection = new \ReflectionClass(ContactTypeForm::class);
    $method = $reflection->getMethod('actions');
    $method->setAccessible(TRUE);

    $actions = $method->invoke($this->form, [], $form_state);

    $this->assertIsArray($actions);
    $this->assertArrayHasKey('submit', $actions);
    $this->assertArrayHasKey('delete', $actions);

    $expected_submit = $this->getStringTranslationStub()->translate('Save contact type');
    $expected_delete = $this->getStringTranslationStub()->translate('Delete contact type');

    $this->assertEquals($expected_submit, $actions['submit']['#value']);
    $this->assertEquals($expected_delete, $actions['delete']['#value']);
  }

  /**
   * Test the save method with new entity.
   *
   * @covers ::save
   */
  public function testSaveNewEntity(): void {
    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->expects($this->exactly(6))
      ->method('getValue')
      ->willReturnMap([
        ['start_date_label', 'Start Date Label'],
        ['start_date_description', 'Start Date Description'],
        ['end_date_label', 'End Date Label'],
        ['end_date_description', 'End Date Description'],
        ['description', 'Test Description'],
        ['label', 'Test Label'],
      ]);

    $this->entity->expects($this->exactly(4))
      ->method('set');

    $this->entity->expects($this->once())
      ->method('id')
      ->willReturn('test_id');
    $this->entity->expects($this->any())
      ->method('label')
      ->willReturn('Test Label');
    $this->entity->expects($this->once())
      ->method('save')
      ->willReturn(self::SAVED_NEW);

    $url = $this->createMock('Drupal\Core\Url');
    $this->entity->expects($this->once())
      ->method('toUrl')
      ->with('collection')
      ->willReturn($url);

    $form_state->expects($this->once())
      ->method('setRedirectUrl')
      ->with($url);

    $this->messenger->expects($this->once())
      ->method('addStatus')
      ->with($this->getStringTranslationStub()->translate('The contact type %name has been added.', ['%name' => 'Test Label']));

    $result = $this->form->save([], $form_state);
    $this->assertEquals(self::SAVED_NEW, $result);
  }

  /**
   * Test the save method with existing entity.
   *
   * @covers ::save
   */
  public function testSaveExistingEntity(): void {
    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->expects($this->exactly(6))
      ->method('getValue')
      ->willReturnMap([
        ['start_date_label', 'Updated Start Label'],
        ['start_date_description', 'Updated Start Description'],
        ['end_date_label', 'Updated End Label'],
        ['end_date_description', 'Updated End Description'],
        ['description', 'Updated Description'],
        ['label', 'Updated Label'],
      ]);

    $this->entity->expects($this->any())
      ->method('id')
      ->willReturn('test_id');
    $this->entity->expects($this->any())
      ->method('label')
      ->willReturn('Updated Label');
    $this->entity->expects($this->once())
      ->method('save')
      ->willReturn(self::SAVED_UPDATED);

    $url = $this->createMock('Drupal\Core\Url');
    $this->entity->expects($this->once())
      ->method('toUrl')
      ->with('collection')
      ->willReturn($url);

    $form_state->expects($this->once())
      ->method('setRedirectUrl')
      ->with($url);

    $this->messenger->expects($this->once())
      ->method('addStatus')
      ->with($this->getStringTranslationStub()->translate('The contact type %name has been updated.', ['%name' => 'Updated Label']));

    $result = $this->form->save([], $form_state);
    $this->assertEquals(self::SAVED_UPDATED, $result);
  }

}
