<?php

declare(strict_types=1);

namespace Drupal\crm\Plugin\Field\FieldFormatter;

use Drupal\Core\Field\Attribute\FieldFormatter;
use Drupal\Core\Field\FieldItemListInterface;
use Drupal\Core\Field\FormatterBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * The age formatter.
 */
#[FieldFormatter(
  id: "age",
  label: new TranslatableMarkup("Age"),
  description: new TranslatableMarkup("Display the age of the entity."),
  field_types: ["integer"]
)]
class AgeFormatter extends FormatterBase {

  /**
   * {@inheritdoc}
   */
  public function viewElements(FieldItemListInterface $items, $langcode) {
    $elements = [];

    foreach ($items as $delta => $item) {
      // $item->value is an integer representing the number of days
      $days = (int) $item->value;

      // Convert days to years, months for display.
      $years = floor($days / 365);
      $remaining_days_after_years = $days % 365;
      $months = floor($remaining_days_after_years / 30);
      $remaining_days = $remaining_days_after_years % 30;

      // Optionally read formatter settings here.
      $granularity = $this->getSetting('granularity') ?? 'years';

      $output = match ($granularity) {
        'years' => $years . ' ' . ($years == 1 ? 'year' : 'years'),
        'months' => ($years * 12 + $months) . ' ' . (($years * 12 + $months) == 1 ? 'month' : 'months'),
        'days' => $days . ' ' . ($days == 1 ? 'day' : 'days'),
        default => trim(
          ($years > 0 ? $years . ' ' . ($years == 1 ? 'year' : 'years') . ', ' : '') .
          ($months > 0 ? $months . ' ' . ($months == 1 ? 'month' : 'months') . ', ' : '') .
          ($remaining_days > 0 ? $remaining_days . ' ' . ($remaining_days == 1 ? 'day' : 'days') : ''),
          ', '
        ),
      };

      $elements[$delta] = ['#markup' => $output];
    }

    return $elements;
  }

  /**
   * {@inheritdoc}
   */
  public static function defaultSettings() {
    return ['granularity' => 'years'] + parent::defaultSettings();
  }

  /**
   * {@inheritdoc}
   */
  public function settingsForm(array $form, FormStateInterface $form_state) {
    return [
      'granularity' => [
        '#title' => $this->t('Granularity'),
        '#type' => 'select',
        '#options' => [
          'years' => $this->t('Years'),
          'months' => $this->t('Months'),
          'days' => $this->t('Days'),
          'full' => $this->t('Full (years, months, days)'),
        ],
        '#default_value' => $this->getSetting('granularity'),
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function settingsSummary() {
    return [t('Granularity: @g', ['@g' => $this->getSetting('granularity')])];
  }

}
