<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel\Form;

use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\crm\Form\ContactDetailTypeForm;
use Drupal\crm\Entity\ContactDetailType;
use Drupal\Core\Form\FormState;

/**
 * Kernel tests for the ContactDetailTypeForm.
 *
 * @group crm
 * @covers \Drupal\crm\Form\ContactDetailTypeForm
 */
class ContactDetailTypeFormTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'comment',
    'datetime',
    'name',
    'telephone',
    'address',
  ];

  /**
   * The form under test.
   *
   * @var \Drupal\crm\Form\ContactDetailTypeForm
   */
  protected $form;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The contact detail type storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $contactDetailTypeStorage;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_contact_detail_type');
    $this->installConfig(['crm']);

    $this->entityTypeManager = $this->container->get('entity_type.manager');
    $this->contactDetailTypeStorage = $this->entityTypeManager->getStorage('crm_contact_detail_type');

    $this->form = ContactDetailTypeForm::create($this->container);
    $this->form->setModuleHandler($this->container->get('module_handler'));
  }

  /**
   * Tests form creation and dependency injection.
   */
  public function testFormCreation(): void {
    $this->assertInstanceOf(ContactDetailTypeForm::class, $this->form);
  }

  /**
   * Tests building the form with a new contact detail type.
   */
  public function testBuildFormWithNewContactDetailType(): void {
    $contact_detail_type = ContactDetailType::create([
      'id' => 'test_type',
      'label' => 'Test Contact Detail Type',
    ]);

    $this->form->setEntity($contact_detail_type);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    $this->assertIsArray($form);
    $this->assertArrayHasKey('label', $form);
    $this->assertArrayHasKey('id', $form);
    $this->assertArrayHasKey('description', $form);

    // Test form field properties.
    $this->assertEquals('textfield', $form['label']['#type']);
    $this->assertEquals('Test Contact Detail Type', $form['label']['#default_value']);
    $this->assertTrue($form['label']['#required']);

    $this->assertEquals('machine_name', $form['id']['#type']);
    $this->assertEquals('test_type', $form['id']['#default_value']);

    $this->assertEquals('textarea', $form['description']['#type']);
    $this->assertEquals('', $form['description']['#default_value']);
  }

  /**
   * Tests building the form with an existing contact detail type.
   */
  public function testBuildFormWithExistingContactDetailType(): void {
    $contact_detail_type = ContactDetailType::create([
      'id' => 'existing_type',
      'label' => 'Existing Contact Detail Type',
      'description' => 'This is an existing contact detail type.',
    ]);
    $contact_detail_type->save();

    $this->form->setEntity($contact_detail_type);
    $this->form->setOperation('edit');

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    $this->assertIsArray($form);
    $this->assertArrayHasKey('#title', $form);

    // Test that the title is set for edit operation.
    $this->assertStringContainsString('Edit', $form['#title']->getUntranslatedString());
    $this->assertStringContainsString('%label', $form['#title']->getUntranslatedString());

    // Test that the title contains the expected pattern.
    $expected_title = $this->container->get('string_translation')->translate('Edit %label CRM contact detail type', ['%label' => 'Existing Contact Detail Type']);
    $this->assertEquals($expected_title, $form['#title']);

    // Test that form fields have correct default values.
    $this->assertEquals('Existing Contact Detail Type', $form['label']['#default_value']);
    $this->assertEquals('existing_type', $form['id']['#default_value']);
    $this->assertEquals('This is an existing contact detail type.', $form['description']['#default_value']);
  }

  /**
   * Tests saving a new contact detail type.
   */
  public function testSaveNewContactDetailType(): void {
    $contact_detail_type = ContactDetailType::create([
      'id' => 'new_test_type',
      'label' => 'New Test Contact Detail Type',
      'description' => 'A new test contact detail type.',
    ]);

    $this->form->setEntity($contact_detail_type);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);
    $result = $this->form->save($form, $form_state);

    $this->assertEquals(SAVED_NEW, $result);

    // Verify the entity was saved.
    $saved_entity = $this->contactDetailTypeStorage->load('new_test_type');
    $this->assertNotNull($saved_entity);
    $this->assertEquals('New Test Contact Detail Type', $saved_entity->label());
    $this->assertEquals('A new test contact detail type.', $saved_entity->getDescription());

    // Verify redirect was set.
    $redirect = $form_state->getRedirect();
    $this->assertEquals('entity.crm_contact_detail_type.collection', $redirect->getRouteName());
  }

  /**
   * Tests updating an existing contact detail type.
   */
  public function testSaveExistingContactDetailType(): void {
    // Create and save initial contact detail type.
    $contact_detail_type = ContactDetailType::create([
      'id' => 'update_test_type',
      'label' => 'Original Label',
      'description' => 'Original description.',
    ]);
    $contact_detail_type->save();

    // Update the entity.
    $contact_detail_type->set('label', 'Updated Label');
    $contact_detail_type->set('description', 'Updated description.');

    $this->form->setEntity($contact_detail_type);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);
    $result = $this->form->save($form, $form_state);

    $this->assertEquals(SAVED_UPDATED, $result);

    // Reload and verify the entity was updated.
    $this->contactDetailTypeStorage->resetCache();
    $updated_entity = $this->contactDetailTypeStorage->load('update_test_type');

    $this->assertEquals('Updated Label', $updated_entity->label());
    $this->assertEquals('Updated description.', $updated_entity->getDescription());

    // Verify redirect was set.
    $redirect = $form_state->getRedirect();
    $this->assertEquals('entity.crm_contact_detail_type.collection', $redirect->getRouteName());
  }

  /**
   * Tests form validation with invalid machine name.
   */
  public function testFormValidationWithInvalidMachineName(): void {
    $contact_detail_type = ContactDetailType::create([
      'id' => 'invalid_machine_name!',
      'label' => 'Test Type',
    ]);

    $this->form->setEntity($contact_detail_type);

    $form_state = new FormState();
    // Set form values to simulate user input with invalid machine name.
    $form_state->setValues([
      'label' => 'Test Type',
      'id' => 'invalid_machine_name!',
      'description' => 'Test description',
    ]);

    $form = $this->form->buildForm([], $form_state);

    // Trigger validation by calling validateForm.
    $this->form->validateForm($form, $form_state);

    // Check that the machine name field
    // configuration includes proper validation.
    $this->assertEquals([ContactDetailType::class, 'load'], $form['id']['#machine_name']['exists']);
    $this->assertArrayHasKey('#machine_name', $form['id']);
  }

  /**
   * Tests form with existing contact detail type ID collision.
   */
  public function testFormWithExistingIdCollision(): void {
    // Create an existing contact detail type.
    $existing_type = ContactDetailType::create([
      'id' => 'existing_id',
      'label' => 'Existing Type',
    ]);
    $existing_type->save();

    // Try to create another with the same ID.
    $new_type = ContactDetailType::create([
      'id' => 'existing_id',
      'label' => 'New Type',
    ]);

    $this->form->setEntity($new_type);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // The machine name field should have the exists callback configured.
    $this->assertEquals([ContactDetailType::class, 'load'], $form['id']['#machine_name']['exists']);
  }

  /**
   * Tests form actions for new entity.
   */
  public function testActionsForNewEntity(): void {
    $contact_detail_type = ContactDetailType::create([
      'id' => 'new_type',
      'label' => 'New Type',
    ]);

    $this->form->setEntity($contact_detail_type);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Use Reflection to access protected method.
    $reflection = new \ReflectionClass(ContactDetailTypeForm::class);
    $method = $reflection->getMethod('actions');
    $method->setAccessible(TRUE);

    $actions = $method->invoke($this->form, $form, $form_state);

    $this->assertIsArray($actions);
    $this->assertArrayHasKey('submit', $actions);

    // Test that the submit button has the correct value.
    $this->assertEquals('Save CRM contact detail type', $actions['submit']['#value']->getUntranslatedString());

    // For bundle entity forms, delete action may exist but
    // should be disabled for new entities.
    if (isset($actions['delete'])) {
      $this->assertEquals('Delete CRM contact detail type', $actions['delete']['#value']->getUntranslatedString());
    }
  }

  /**
   * Tests form actions for existing entity.
   */
  public function testActionsForExistingEntity(): void {
    $contact_detail_type = ContactDetailType::create([
      'id' => 'existing_type',
      'label' => 'Existing Type',
    ]);
    $contact_detail_type->save();

    $this->form->setEntity($contact_detail_type);

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Use Reflection to access protected method.
    $reflection = new \ReflectionClass(ContactDetailTypeForm::class);
    $method = $reflection->getMethod('actions');
    $method->setAccessible(TRUE);

    $actions = $method->invoke($this->form, $form, $form_state);

    $this->assertIsArray($actions);
    $this->assertArrayHasKey('submit', $actions);
    $this->assertArrayHasKey('delete', $actions);
  }

}
