<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel;

use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\ContactDetail;
use Drupal\crm\Entity\Relationship;
use Drupal\crm\Entity\UserContact;
use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;

/**
 * Tests crm user contact mapping.
 *
 * @group crm
 */
class HookContactDeleteTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'address',
    'comment',
    'crm',
    'datetime',
    'name',
    'telephone',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installSchema('user', ['users_data']);
    $this->installEntitySchema('comment');
    $this->installSchema('comment', ['comment_entity_statistics']);
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_relationship');
    $this->installEntitySchema('crm_user_contact');

    $this->installConfig(['crm', 'name']);

  }

  /**
   * Verify the mapping is deleted when a user is deleted.
   */
  public function testContactDeleteWithUserMapping(): void {

    $organization_email = ContactDetail::create([
      'email' => 'organization@example.local',
      'bundle' => 'email',
    ]);
    $organization_email->save();
    $organization_telephone = ContactDetail::create([
      'telephone' => '(555) 555 - 5555',
      'bundle' => 'telephone',
    ]);
    $organization_telephone->save();
    $organization_address = ContactDetail::create([
      'address' => [
        'city' => 'New Lenox',
      ],
      'bundle' => 'address',
    ]);
    $organization_address->save();

    $organization = Contact::create([
      'name' => 'Organization',
      'bundle' => 'organization',
      'emails' => [$organization_email],
      'telephones' => [$organization_telephone],
      'addresses' => [$organization_address],
    ]);
    $organization->save();

    // Create a user.
    $user_1 = $this->drupalCreateUser();
    $user_1->save();

    $person_1_email = ContactDetail::create([
      'email' => 'person_1@example.local',
      'bundle' => 'email',
    ]);
    $person_1_email->save();
    $person_1_telephone = ContactDetail::create([
      'telephone' => '(555) 555 - 5555',
      'bundle' => 'telephone',
    ]);
    $person_1_telephone->save();
    $person_1_address = ContactDetail::create([
      'address' => [
        'city' => 'New Lenox',
      ],
      'bundle' => 'address',
    ]);
    $person_1_address->save();
    $person_1 = Contact::create([
      'name' => 'Test person',
      'bundle' => 'person',
      'full_name' => [
        'given_name' => 'Test',
        'family_name' => 'UserContact 1',
      ],
      'emails' => [$person_1_email],
      'telephones' => [$person_1_telephone],
      'addresses' => [$person_1_address],
    ]);
    $person_1->save();

    $crm_user_1 = UserContact::create([
      'user' => $user_1->id(),
      'crm_contact' => $person_1->id(),
    ]);
    $crm_user_1->save();

    $user_2 = $this->drupalCreateUser();
    $user_2->save();

    $person_2_email = ContactDetail::create([
      'email' => 'person_2@example.local',
      'bundle' => 'email',
    ]);
    $person_2_email->save();
    $person_2_telephone = ContactDetail::create([
      'telephone' => '(555) 555 - 5555',
      'bundle' => 'telephone',
    ]);
    $person_2_telephone->save();
    $person_2_address = ContactDetail::create([
      'address' => [
        'city' => 'New Lenox',
      ],
      'bundle' => 'address',
    ]);
    $person_2_address->save();
    $person_2 = Contact::create([
      'name' => 'Test person',
      'bundle' => 'person',
      'full_name' => [
        'given_name' => 'Test',
        'family_name' => 'UserContact 2',
      ],
      'emails' => [$person_2_email],
      'telephones' => [$person_2_telephone],
      'addresses' => [$person_2_address],
    ]);
    $person_2->save();

    $crm_user_2 = UserContact::create([
      'user' => $user_2->id(),
      'crm_contact' => $person_2->id(),
    ]);
    $crm_user_2->save();

    $relationship_siblings = Relationship::create([
      'bundle' => 'siblings',
      'contacts' => [$person_1, $person_2],
    ]);
    $relationship_siblings->save();

    $relationship_employee_1 = Relationship::create([
      'bundle' => 'employee',
      'contacts' => [$person_1, $organization],
    ]);
    $relationship_employee_1->save();

    $relationship_employee_2 = Relationship::create([
      'bundle' => 'employee',
      'contacts' => [$person_2, $organization],
    ]);
    $relationship_employee_2->save();

    $crm_users = UserContact::loadMultiple();
    $this->assertCount(2, $crm_users, 'Two crm_user exists.');

    $crm_contacts = Contact::loadMultiple();
    $this->assertCount(3, $crm_contacts, 'Three crm_contacts exists.');

    $crm_contact_details = ContactDetail::loadMultiple();
    $this->assertCount(9, $crm_contact_details, 'Nine crm_contact_details exists.');

    $crm_relationships = Relationship::loadMultiple();
    $this->assertCount(3, $crm_relationships, 'Three crm_relationships exists.');

    // Delete the first user.
    $person_1->delete();

    // Reset the user cache to ensure we are working with the latest data.
    $crm_users = UserContact::loadMultiple();
    $this->assertCount(1, $crm_users, 'One crm_user entity remains after deleting the user.');

    $crm_user = reset($crm_users);
    $this->assertEquals($user_2->id(), $crm_user->getUserId(), 'The remaining crm_user entity is linked to the second user.');
    $this->assertEquals($person_2->id(), $crm_user->getContactId(), 'The remaining crm_user entity is linked to the second contact.');

    $crm_contacts = Contact::loadMultiple();
    $this->assertCount(2, $crm_contacts, 'Two crm_contacts exists.');

    $crm_contact_details = ContactDetail::loadMultiple();
    $this->assertCount(6, $crm_contact_details, 'Six crm_contact_details exists.');

    $crm_relationships = Relationship::loadMultiple();

    $this->assertCount(1, $crm_relationships, 'One crm_relationships exists.');
  }

}
