<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel;

use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\ContactDetail;

/**
 * Tests crm user contact mapping.
 *
 * @group crm
 */
class UserContactCreationTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'address',
    'comment',
    'crm',
    'datetime',
    'name',
    'telephone',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('comment');
    $this->installSchema('comment', ['comment_entity_statistics']);
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_user_contact');

    $this->installConfig(['crm', 'name']);

  }

  /**
   * Tests creating a contact when the user is created.
   */
  public function testCreateContact(): void {
    $settings = $this->config('crm.user_contact.settings');
    $settings->set('auto_create_crm_user_contact', TRUE);
    $settings->save();

    // Get all crm_user entities.
    $query = \Drupal::entityQuery('crm_user_contact');
    $query->accessCheck(FALSE);
    $pre_count = $query->count()->execute();

    $user = $this->createUser([], 'a_user', FALSE, ['mail' => 'test@drupal.local']);

    $query = \Drupal::entityQuery('crm_user_contact');
    $query->accessCheck(FALSE);
    $post_count = $query->count()->execute();

    $this->assertEquals($pre_count + 1, $post_count, 'A crm_user entity was created when the user was created.');

    $query = $this->entityTypeManager->getStorage('crm_contact_detail')->getQuery();
    $query->condition('email', 'test@drupal.local')
      ->condition('bundle', 'email');
    $contact_detail = $query->accessCheck(FALSE)->execute();

    $this->assertCount(1, $contact_detail, 'A contact detail was created for the user email.');

  }

  /**
   * Tests creating a contact when the user is created.
   */
  public function testCreateContactMatch(): void {
    $settings = $this->config('crm.user_contact.settings');
    $settings->set('auto_create_crm_user_contact', TRUE);
    $settings->set('auto_create_lookup_contact', TRUE);
    $settings->save();

    // Get all crm_user entities.
    $query = \Drupal::entityQuery('crm_user_contact');
    $query->accessCheck(FALSE);
    $pre_count = $query->count()->execute();

    $contact_email = ContactDetail::create([
      'bundle' => 'email',
      'email' => 'test@drupal.local',
      'type' => 'main',
    ]);
    $contact_email->save();
    $contact = Contact::create([
      'bundle' => 'person',
      'full_name' => ['given' => 'Test User'],
      'emails' => [$contact_email],
    ]);
    $contact->save();

    $query = \Drupal::entityQuery('crm_contact');
    $query->accessCheck(FALSE);
    $pre_contact_count = $query->count()->execute();

    $user = $this->createUser([], 'a_user', FALSE, ['mail' => 'test@drupal.local']);

    $query = \Drupal::entityQuery('crm_user_contact');
    $query->accessCheck(FALSE);
    $post_count = $query->count()->execute();

    $this->assertEquals($pre_count + 1, $post_count, 'A crm_user entity was created when the user was created.');

    $query = $this->entityTypeManager->getStorage('crm_contact_detail')->getQuery();
    $query->condition('email', 'test@drupal.local')
      ->condition('bundle', 'email');
    $contact_detail = $query->accessCheck(FALSE)->execute();

    $this->assertCount(1, $contact_detail, 'A contact detail was created for the user email.');

    $query = \Drupal::entityQuery('crm_contact');
    $query->accessCheck(FALSE);
    $post_contact_count = $query->count()->execute();

    $this->assertEquals($pre_contact_count, $post_contact_count, 'A contact was mapped for the user.');

  }

}
