<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Unit\Hook;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Form\FormStateInterface;
use Drupal\crm\Hook\UserHooks;
use Drupal\Tests\UnitTestCase;
use Drupal\user\UserDataInterface;
use Drupal\user\UserInterface;

/**
 * Tests UserHooks functionality.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Hook\UserHooks
 */
class UserHooksTest extends UnitTestCase {

  /**
   * Mock user data service.
   *
   * @var \Drupal\user\UserDataInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $mockUserData;

  /**
   * Mock form state.
   *
   * @var \Drupal\Core\Form\FormStateInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $mockFormState;

  /**
   * Mock user entity.
   *
   * @var \Drupal\user\UserInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $mockUser;

  /**
   * Mock form object.
   *
   * @var \Drupal\Core\Entity\EntityFormInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $mockFormObject;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create mock services.
    $this->mockUserData = $this->createMock(UserDataInterface::class);
    $this->mockFormState = $this->createMock(FormStateInterface::class);
    $this->mockUser = $this->createMock(UserInterface::class);
    $this->mockFormObject = $this->getMockBuilder('Drupal\Core\Entity\EntityFormInterface')
      ->disableOriginalConstructor()
      ->getMock();

    // Set up container with mock services.
    $container = new ContainerBuilder();
    $container->set('user.data', $this->mockUserData);
    \Drupal::setContainer($container);

    // Set up form state mock to return form object.
    $this->mockFormState
      ->method('getFormObject')
      ->willReturn($this->mockFormObject);

    // Set up form object to return user entity.
    $this->mockFormObject
      ->method('getEntity')
      ->willReturn($this->mockUser);

    // Set up user mock to return ID.
    $this->mockUser
      ->method('id')
      ->willReturn('123');
  }

  /**
   * Tests userFormAlterSubmit with a valid name format.
   *
   * @covers ::userFormAlterSubmit
   */
  public function testUserFormAlterSubmitWithValidFormat(): void {
    // Set up form state to return a specific name format value.
    $this->mockFormState
      ->expects($this->once())
      ->method('getValue')
      ->with('name_format', NULL)
      ->willReturn('test_format');

    // Expect the user data service to set the value.
    $this->mockUserData
      ->expects($this->once())
      ->method('set')
      ->with('crm', '123', 'name_format', 'test_format');

    // Ensure delete is not called.
    $this->mockUserData
      ->expects($this->never())
      ->method('delete');

    // Create a dummy form array.
    $form = [];

    // Call the static method.
    UserHooks::userFormAlterSubmit($form, $this->mockFormState);
  }

  /**
   * Tests userFormAlterSubmit with _none option to clear format.
   *
   * @covers ::userFormAlterSubmit
   */
  public function testUserFormAlterSubmitWithNoneOption(): void {
    // Set up form state to return _none.
    $this->mockFormState
      ->expects($this->once())
      ->method('getValue')
      ->with('name_format', NULL)
      ->willReturn('_none');

    // Expect the user data service to delete the value.
    $this->mockUserData
      ->expects($this->once())
      ->method('delete')
      ->with('crm', '123', 'name_format');

    // Ensure set is not called.
    $this->mockUserData
      ->expects($this->never())
      ->method('set');

    // Create a dummy form array.
    $form = [];

    // Call the static method.
    UserHooks::userFormAlterSubmit($form, $this->mockFormState);
  }

  /**
   * Tests userFormAlterSubmit with null value.
   *
   * @covers ::userFormAlterSubmit
   */
  public function testUserFormAlterSubmitWithNullValue(): void {
    // Set up form state to return null.
    $this->mockFormState
      ->expects($this->once())
      ->method('getValue')
      ->with('name_format', NULL)
      ->willReturn(NULL);

    // Expect the user data service to set null (which should still work).
    $this->mockUserData
      ->expects($this->once())
      ->method('set')
      ->with('crm', '123', 'name_format', NULL);

    // Ensure delete is not called for null values.
    $this->mockUserData
      ->expects($this->never())
      ->method('delete');

    // Create a dummy form array.
    $form = [];

    // Call the static method.
    UserHooks::userFormAlterSubmit($form, $this->mockFormState);
  }

  /**
   * Tests userFormAlterSubmit with empty string value.
   *
   * @covers ::userFormAlterSubmit
   */
  public function testUserFormAlterSubmitWithEmptyString(): void {
    // Set up form state to return empty string.
    $this->mockFormState
      ->expects($this->once())
      ->method('getValue')
      ->with('name_format', NULL)
      ->willReturn('');

    // Expect the user data service to set empty string.
    $this->mockUserData
      ->expects($this->once())
      ->method('set')
      ->with('crm', '123', 'name_format', '');

    // Ensure delete is not called.
    $this->mockUserData
      ->expects($this->never())
      ->method('delete');

    // Create a dummy form array.
    $form = [];

    // Call the static method.
    UserHooks::userFormAlterSubmit($form, $this->mockFormState);
  }

  /**
   * Tests that the method properly handles the form array by reference.
   *
   * @covers ::userFormAlterSubmit
   */
  public function testUserFormAlterSubmitFormByReference(): void {
    // Set up form state to return a valid format.
    $this->mockFormState
      ->method('getValue')
      ->with('name_format', NULL)
      ->willReturn('test_format');

    // Set up user data expectations.
    $this->mockUserData
      ->expects($this->once())
      ->method('set')
      ->with('crm', '123', 'name_format', 'test_format');

    // Create a form array with some data.
    $form = [
      'original_data' => 'should_remain',
    ];

    // Call the static method.
    UserHooks::userFormAlterSubmit($form, $this->mockFormState);

    // Verify the form array is unchanged (method doesn't modify it).
    $this->assertEquals(['original_data' => 'should_remain'], $form);
  }

}
