<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Unit\Form;

use Drupal\Tests\UnitTestCase;
use Drupal\crm\Form\RelationshipTypeForm;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Messenger\MessengerInterface;

/**
 * Relationship type form test.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Form\RelationshipTypeForm
 * @uses \Drupal\crm\Form\RelationshipTypeForm
 */
class RelationshipTypeFormTest extends UnitTestCase {

  const SAVED_NEW = 1;
  const SAVED_UPDATED = 2;

  /**
   * The form.
   *
   * @var \Drupal\crm\Form\RelationshipTypeForm
   */
  protected $form;

  /**
   * The messenger.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $messenger;


  /**
   * The entity type bundle info.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeBundleInfo;

  /**
   * The entity.
   *
   * @var \Drupal\crm\Entity\RelationshipType|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entity;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $container = new ContainerBuilder();

    $string_translation = $this->getStringTranslationStub();
    $container->set('string_translation', $string_translation);

    $this->messenger = $this->createMock(MessengerInterface::class);
    $container->set('messenger', $this->messenger);

    $this->entityTypeBundleInfo = $this->createMock('Drupal\Core\Entity\EntityTypeBundleInfoInterface');
    $container->set('entity_type.bundle.info', $this->entityTypeBundleInfo);

    \Drupal::setContainer($container);

    $this->form = RelationshipTypeForm::create($container);

    $this->entity = $this->createMock('Drupal\crm\Entity\RelationshipType');
    $this->form->setEntity($this->entity);

  }

  /**
   * Tests the create method.
   *
   * @covers ::create
   */
  public function testCreate() {
    $container = \Drupal::getContainer();
    $form = RelationshipTypeForm::create($container);

    $this->assertInstanceOf(RelationshipTypeForm::class, $form);
  }

  /**
   * Test the constructor.
   *
   * @covers ::__construct
   */
  public function testConstructor() {
    $form = new RelationshipTypeForm(
      $this->entityTypeBundleInfo,
    );
    $this->assertInstanceOf(RelationshipTypeForm::class, $form);
  }

  /**
   * Test the form id.
   *
   * @covers ::getFormId
   */
  public function testGetFormId() {
    $entity_type = $this->createMock('Drupal\Core\Entity\EntityTypeInterface');
    $entity_type->expects($this->once())
      ->method('hasKey')
      ->with('bundle')
      ->willReturn(FALSE);
    $this->entity->expects($this->once())
      ->method('getEntityTypeId')
      ->willReturn('crm_relationship_type');
    $this->entity->expects($this->once())
      ->method('getEntityType')
      ->willReturn($entity_type);

    $this->form->setOperation('edit');

    $this->assertEquals('crm_relationship_type_edit_form', $this->form->getFormId());
  }

  /**
   * Test the form build.
   *
   * @covers ::form
   */
  public function testForm(): void {
    $entity_type = $this->createMock('Drupal\Core\Entity\EntityTypeInterface');
    $entity_type->expects($this->once())
      ->method('getKey')
      ->with('id')
      ->willReturn('id');

    $this->entity->expects($this->once())
      ->method('getEntityType')
      ->willReturn($entity_type);
    $this->entity->expects($this->any())
      ->method('label')
      ->willReturn('Test Relationship Type');
    $this->entity->expects($this->any())
      ->method('id')
      ->willReturn('test_relationship_type');
    $this->entity->expects($this->any())
      ->method('get')
      ->willReturnMap([
        ['description', 'Test description'],
        ['asymmetric', TRUE],
        ['label_a', 'Contact A Label'],
        ['contact_type_a', 'person'],
        ['label_b', 'Contact B Label'],
        ['contact_type_b', 'organization'],
      ]);
    $this->entity->expects($this->any())
      ->method('isNew')
      ->willReturn(FALSE);

    $this->entityTypeBundleInfo->expects($this->exactly(2))
      ->method('getBundleInfo')
      ->with('crm_contact')
      ->willReturn([
        'person' => ['label' => 'Person'],
        'household' => ['label' => 'Household'],
        'organization' => ['label' => 'Organization'],
      ]);

    $form_state = $this->createMock('Drupal\Core\Form\FormStateInterface');
    $form = $this->form->form([], $form_state);

    $this->assertIsArray($form);
    $this->assertArrayHasKey('label', $form);
    $this->assertArrayHasKey('id', $form);
    $this->assertArrayHasKey('description', $form);
    $this->assertArrayHasKey('asymmetric', $form);
    $this->assertArrayHasKey('label_a', $form);
    $this->assertArrayHasKey('contact_type_a', $form);
    $this->assertArrayHasKey('label_b', $form);
    $this->assertArrayHasKey('contact_type_b', $form);

    // Test label field.
    $this->assertEquals('textfield', $form['label']['#type']);
    $this->assertEquals('Test Relationship Type', $form['label']['#default_value']);
    $this->assertTrue($form['label']['#required']);

    // Test id field (machine name).
    $this->assertEquals('machine_name', $form['id']['#type']);
    $this->assertEquals('test_relationship_type', $form['id']['#default_value']);

    // Test description field.
    $this->assertEquals('textarea', $form['description']['#type']);
    $this->assertEquals('Test description', $form['description']['#default_value']);

    // Test asymmetric checkbox.
    $this->assertEquals('checkbox', $form['asymmetric']['#type']);
    $this->assertEquals(TRUE, $form['asymmetric']['#default_value']);

    // Test label_a field.
    $this->assertEquals('textfield', $form['label_a']['#type']);
    $this->assertEquals('Contact A Label', $form['label_a']['#default_value']);
    $this->assertTrue($form['label_a']['#required']);

    // Test contact_type_a field.
    $this->assertEquals('select', $form['contact_type_a']['#type']);
    $this->assertEquals('person', $form['contact_type_a']['#default_value']);
    $this->assertTrue($form['contact_type_a']['#required']);
    $expected_options = [
      'person' => 'Person',
      'household' => 'Household',
      'organization' => 'Organization',
    ];
    $this->assertEquals($expected_options, $form['contact_type_a']['#options']);

    // Test label_b field (visible when asymmetric).
    $this->assertEquals('textfield', $form['label_b']['#type']);
    $this->assertEquals('Contact B Label', $form['label_b']['#default_value']);
    $this->assertTrue($form['label_b']['#required']);
    $this->assertArrayHasKey('#states', $form['label_b']);

    // Test contact_type_b field (visible when asymmetric).
    $this->assertEquals('select', $form['contact_type_b']['#type']);
    $this->assertEquals('organization', $form['contact_type_b']['#default_value']);
    $this->assertTrue($form['contact_type_b']['#required']);
    $this->assertEquals($expected_options, $form['contact_type_b']['#options']);
    $this->assertArrayHasKey('#states', $form['contact_type_b']);
  }

  /**
   * Test the form build with edit operation.
   *
   * @covers ::form
   */
  public function testFormWithEditOperation(): void {
    $entity_type = $this->createMock('Drupal\Core\Entity\EntityTypeInterface');
    $entity_type->expects($this->once())
      ->method('getKey')
      ->with('id')
      ->willReturn('id');

    $this->entity->expects($this->once())
      ->method('getEntityType')
      ->willReturn($entity_type);
    $this->entity->expects($this->any())
      ->method('label')
      ->willReturn('Test Relationship Type');
    $this->entity->expects($this->any())
      ->method('id')
      ->willReturn('test_relationship_type');
    $this->entity->expects($this->any())
      ->method('get')
      ->willReturnMap([
        ['description', 'Test description'],
        ['asymmetric', FALSE],
        ['label_a', 'Contact A Label'],
        ['contact_type_a', 'person'],
        ['label_b', 'Contact B Label'],
        ['contact_type_b', 'organization'],
      ]);
    $this->entity->expects($this->any())
      ->method('isNew')
      ->willReturn(FALSE);

    $this->entityTypeBundleInfo->expects($this->exactly(2))
      ->method('getBundleInfo')
      ->with('crm_contact')
      ->willReturn([
        'person' => ['label' => 'Person'],
        'household' => ['label' => 'Household'],
        'organization' => ['label' => 'Organization'],
      ]);

    $this->form->setOperation('edit');

    $form_state = $this->createMock('Drupal\Core\Form\FormStateInterface');
    $form = $this->form->form([], $form_state);

    $this->assertArrayHasKey('#title', $form);
    $expected_title = $this->getStringTranslationStub()->translate('Edit %label crm relationship type', ['%label' => 'Test Relationship Type']);
    $this->assertEquals($expected_title, $form['#title']);
  }

  /**
   * Test the getContactTypeOptions method.
   *
   * @covers ::getContactTypeOptions
   */
  public function testGetContactTypeOptions(): void {
    $this->entityTypeBundleInfo->expects($this->once())
      ->method('getBundleInfo')
      ->with('crm_contact')
      ->willReturn([
        'person' => ['label' => 'Person'],
        'household' => ['label' => 'Household'],
        'organization' => ['label' => 'Organization'],
      ]);

    // Use reflection to access the protected method.
    $reflection = new \ReflectionClass(RelationshipTypeForm::class);
    $method = $reflection->getMethod('getContactTypeOptions');
    $method->setAccessible(TRUE);

    $options = $method->invoke($this->form);

    $expected_options = [
      'person' => 'Person',
      'household' => 'Household',
      'organization' => 'Organization',
    ];
    $this->assertEquals($expected_options, $options);
  }

  /**
   * Test the actions method.
   *
   * @covers ::actions
   */
  public function testActions(): void {
    $form_state = $this->createMock('Drupal\Core\Form\FormStateInterface');

    // We need to set up the entity properly to call actions.
    $this->entity->expects($this->any())
      ->method('isNew')
      ->willReturn(FALSE);

    // Use reflection to access the protected method.
    $reflection = new \ReflectionClass(RelationshipTypeForm::class);
    $method = $reflection->getMethod('actions');
    $method->setAccessible(TRUE);

    $actions = $method->invoke($this->form, [], $form_state);

    $this->assertIsArray($actions);
    $this->assertArrayHasKey('submit', $actions);

    $expected_submit = $this->getStringTranslationStub()->translate('Save relationship type');
    $this->assertEquals($expected_submit, $actions['submit']['#value']);
  }

  /**
   * Test the save method with new entity.
   *
   * @covers ::save
   */
  public function testSaveNewEntity(): void {
    $form_state = $this->createMock('Drupal\Core\Form\FormStateInterface');

    $this->entity->expects($this->exactly(2))
      ->method('set')
      ->willReturnCallback(function ($key, $value) {
        static $call_count = 0;
        $call_count++;
        if ($call_count === 1) {
          $this->assertEquals('id', $key);
          $this->assertEquals('test_id', $value);
        }
        elseif ($call_count === 2) {
          $this->assertEquals('label', $key);
          $this->assertEquals('Test Label', $value);
        }
        return $this->entity;
      });
    $this->entity->expects($this->once())
      ->method('id')
      ->willReturn('test_id');
    $this->entity->expects($this->any())
      ->method('label')
      ->willReturn('Test Label');
    $this->entity->expects($this->once())
      ->method('get')
      ->with('asymmetric')
      ->willReturn(TRUE);
    $this->entity->expects($this->once())
      ->method('save')
      ->willReturn(self::SAVED_NEW);

    $url = $this->createMock('Drupal\Core\Url');
    $this->entity->expects($this->once())
      ->method('toUrl')
      ->with('collection')
      ->willReturn($url);

    $form_state->expects($this->once())
      ->method('setRedirectUrl')
      ->with($url);

    $this->messenger->expects($this->once())
      ->method('addStatus')
      ->with($this->getStringTranslationStub()->translate('The crm relationship type %name has been added.', ['%name' => 'Test Label']));

    $result = $this->form->save([], $form_state);
    $this->assertEquals(self::SAVED_NEW, $result);
  }

  /**
   * Test the save method with existing entity.
   *
   * @covers ::save
   */
  public function testSaveExistingEntity(): void {
    $form_state = $this->createMock('Drupal\Core\Form\FormStateInterface');

    $this->entity->expects($this->exactly(2))
      ->method('set')
      ->willReturnCallback(function ($key, $value) {
        static $call_count = 0;
        $call_count++;
        if ($call_count === 1) {
          $this->assertEquals('id', $key);
          $this->assertEquals('updated_id', $value);
        }
        elseif ($call_count === 2) {
          $this->assertEquals('label', $key);
          $this->assertEquals('Updated Label', $value);
        }
        return $this->entity;
      });
    $this->entity->expects($this->once())
      ->method('id')
      ->willReturn('updated_id');
    $this->entity->expects($this->any())
      ->method('label')
      ->willReturn('Updated Label');
    $this->entity->expects($this->once())
      ->method('get')
      ->with('asymmetric')
      ->willReturn(TRUE);
    $this->entity->expects($this->once())
      ->method('save')
      ->willReturn(self::SAVED_UPDATED);

    $url = $this->createMock('Drupal\Core\Url');
    $this->entity->expects($this->once())
      ->method('toUrl')
      ->with('collection')
      ->willReturn($url);

    $form_state->expects($this->once())
      ->method('setRedirectUrl')
      ->with($url);

    $this->messenger->expects($this->once())
      ->method('addStatus')
      ->with($this->getStringTranslationStub()->translate('The crm relationship type %name has been updated.', ['%name' => 'Updated Label']));

    $result = $this->form->save([], $form_state);
    $this->assertEquals(self::SAVED_UPDATED, $result);
  }

  /**
   * Test the save method with symmetric relationship.
   *
   * @covers ::save
   */
  public function testSaveSymmetricRelationship(): void {
    $form_state = $this->createMock('Drupal\Core\Form\FormStateInterface');

    $this->entity->expects($this->exactly(4))
      ->method('set')
      ->willReturnCallback(function ($key, $value) {
        static $call_count = 0;
        $call_count++;
        if ($call_count === 1) {
          $this->assertEquals('id', $key);
          $this->assertEquals('symmetric_id', $value);
        }
        elseif ($call_count === 2) {
          $this->assertEquals('label', $key);
          $this->assertEquals('Symmetric Label', $value);
        }
        elseif ($call_count === 3) {
          $this->assertEquals('label_b', $key);
          $this->assertEquals('Contact A Label', $value);
        }
        elseif ($call_count === 4) {
          $this->assertEquals('contact_type_b', $key);
          $this->assertEquals('person', $value);
        }
        return $this->entity;
      });
    $this->entity->expects($this->once())
      ->method('id')
      ->willReturn('symmetric_id');
    $this->entity->expects($this->any())
      ->method('label')
      ->willReturn('Symmetric Label');
    $this->entity->expects($this->exactly(3))
      ->method('get')
      ->willReturnMap([
        ['asymmetric', FALSE],
        ['label_a', 'Contact A Label'],
        ['contact_type_a', 'person'],
      ]);
    $this->entity->expects($this->once())
      ->method('save')
      ->willReturn(self::SAVED_NEW);

    $url = $this->createMock('Drupal\Core\Url');
    $this->entity->expects($this->once())
      ->method('toUrl')
      ->with('collection')
      ->willReturn($url);

    $form_state->expects($this->once())
      ->method('setRedirectUrl')
      ->with($url);

    $this->messenger->expects($this->once())
      ->method('addStatus')
      ->with($this->getStringTranslationStub()->translate('The crm relationship type %name has been added.', ['%name' => 'Symmetric Label']));

    $result = $this->form->save([], $form_state);
    $this->assertEquals(self::SAVED_NEW, $result);
  }

}
