<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Unit\Form;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\crm\Entity\UserContact;
use Drupal\crm\Form\UserContactForm;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\HttpFoundation\InputBag;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\Display\EntityFormDisplayInterface;

/**
 * Unit tests for the UserContactForm class.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Form\UserContactForm
 * @uses \Drupal\crm\Form\UserContactForm
 */
class UserContactFormTest extends UnitTestCase {

  const SAVED_NEW = 1;
  const SAVED_UPDATED = 2;

  /**
   * The form under test.
   *
   * @var \Drupal\crm\Form\UserContactForm
   */
  protected $form;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $messenger;

  /**
   * The logger service.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $logger;

  /**
   * The user contact entity.
   *
   * @var \Drupal\crm\Entity\UserContact|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entity;

  /**
   * The request object.
   *
   * @var \Symfony\Component\HttpFoundation\Request|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $request;

  /**
   * The time service.
   *
   * @var \Drupal\Component\Datetime\TimeInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $time;

  /**
   * The entity repository service.
   *
   * @var \Drupal\Core\Entity\EntityRepositoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityRepository;

  /**
   * The entity type bundle info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeBundleInfo;

  /**
   * The request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $requestStack;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $container = new ContainerBuilder();

    $string_translation = $this->getStringTranslationStub();
    $container->set('string_translation', $string_translation);

    $this->messenger = $this->createMock(MessengerInterface::class);
    $container->set('messenger', $this->messenger);

    $this->logger = $this->createMock(LoggerChannelInterface::class);
    $logger_factory = $this->createMock('Drupal\Core\Logger\LoggerChannelFactoryInterface');
    $logger_factory->expects($this->any())
      ->method('get')
      ->with('crm')
      ->willReturn($this->logger);
    $container->set('logger.factory', $logger_factory);

    $this->entityRepository = $this->createMock(EntityRepositoryInterface::class);
    $container->set('entity.repository', $this->entityRepository);

    $this->entityTypeBundleInfo = $this->createMock(EntityTypeBundleInfoInterface::class);
    $container->set('entity_type.bundle.info', $this->entityTypeBundleInfo);

    $this->time = $this->createMock(TimeInterface::class);
    $container->set('datetime.time', $this->time);

    $this->requestStack = $this->createMock(RequestStack::class);
    $container->set('request_stack', $this->requestStack);

    \Drupal::setContainer($container);

    // Mock request.
    $this->request = $this->createMock(Request::class);
    $this->request->query = new InputBag();
    $this->requestStack->expects($this->any())
      ->method('getCurrentRequest')
      ->willReturn($this->request);

    $this->form = new UserContactForm($this->entityRepository, $this->entityTypeBundleInfo, $this->time);
    $this->entity = $this->createMock(UserContact::class);
    $this->form->setEntity($this->entity);

  }

  /**
   * Tests the form method with no query parameters.
   *
   * @covers ::form
   */
  public function testFormWithoutQueryParameters(): void {
    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->expects($this->once())
      ->method('get')
      ->with('form_display')
      ->willReturn($this->createMock(EntityFormDisplayInterface::class));

    $this->request->query->set('crm_contact', NULL);
    $this->request->query->set('user', NULL);

    $this->entity->expects($this->never())
      ->method('setContactId');
    $this->entity->expects($this->never())
      ->method('setUserId');

    $this->entity->expects($this->once())
      ->method('isNew')
      ->willReturn(TRUE);

    $entity_type = $this->createMock(EntityTypeInterface::class);
    $entity_type->expects($this->exactly(2))
      ->method('showRevisionUi')
      ->willReturn(FALSE);

    $this->entity->expects($this->exactly(2))
      ->method('getEntityType')
      ->willReturn($entity_type);

    $form = $this->form->form([], $form_state);

    $this->assertIsArray($form);
    $this->assertArrayNotHasKey('crm_contact', $form);
    $this->assertArrayNotHasKey('user', $form);
  }

  /**
   * Tests the form method with crm_contact query parameter.
   *
   * @covers ::form
   */
  public function testFormWithCrmContactParameter(): void {
    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->expects($this->once())
      ->method('get')
      ->with('form_display')
      ->willReturn($this->createMock(EntityFormDisplayInterface::class));

    $this->request->query->set('crm_contact', '123');
    $this->request->query->set('user', NULL);

    $this->entity->expects($this->once())
      ->method('setContactId')
      ->with('123')
      ->willReturnSelf();
    $this->entity->expects($this->once())
      ->method('setUserId')
      ->with(NULL)
      ->willReturnSelf();

    $this->entity->expects($this->once())
      ->method('isNew')
      ->willReturn(TRUE);

    $entity_type = $this->createMock(EntityTypeInterface::class);
    $entity_type->expects($this->exactly(2))
      ->method('showRevisionUi')
      ->willReturn(FALSE);

    $this->entity->expects($this->exactly(2))
      ->method('getEntityType')
      ->willReturn($entity_type);

    $form = $this->form->form([], $form_state);

    $this->assertIsArray($form);
    $this->assertArrayHasKey('crm_contact', $form);
    $this->assertTrue($form['crm_contact']['widget']['#disabled']);
  }

  /**
   * Tests the form method with user query parameter.
   *
   * @covers ::form
   */
  public function testFormWithUserParameter(): void {
    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->expects($this->once())
      ->method('get')
      ->with('form_display')
      ->willReturn($this->createMock(EntityFormDisplayInterface::class));

    $this->request->query->set('crm_contact', NULL);
    $this->request->query->set('user', '456');

    $this->entity->expects($this->once())
      ->method('setContactId')
      ->with(NULL)
      ->willReturnSelf();
    $this->entity->expects($this->once())
      ->method('setUserId')
      ->with('456')
      ->willReturnSelf();

    $this->entity->expects($this->once())
      ->method('isNew')
      ->willReturn(TRUE);

    $entity_type = $this->createMock(EntityTypeInterface::class);
    $entity_type->expects($this->exactly(2))
      ->method('showRevisionUi')
      ->willReturn(FALSE);

    $this->entity->expects($this->exactly(2))
      ->method('getEntityType')
      ->willReturn($entity_type);

    $form = $this->form->form([], $form_state);

    $this->assertIsArray($form);
    $this->assertArrayHasKey('user', $form);
    $this->assertTrue($form['user']['widget']['#disabled']);
  }

  /**
   * Tests the form method with both query parameters.
   *
   * @covers ::form
   */
  public function testFormWithBothParameters(): void {
    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->expects($this->once())
      ->method('get')
      ->with('form_display')
      ->willReturn($this->createMock(EntityFormDisplayInterface::class));

    $this->request->query->set('crm_contact', '123');
    $this->request->query->set('user', '456');

    $this->entity->expects($this->once())
      ->method('setContactId')
      ->with('123')
      ->willReturnSelf();
    $this->entity->expects($this->once())
      ->method('setUserId')
      ->with('456')
      ->willReturnSelf();

    $this->entity->expects($this->once())
      ->method('isNew')
      ->willReturn(TRUE);

    $entity_type = $this->createMock(EntityTypeInterface::class);
    $entity_type->expects($this->exactly(2))
      ->method('showRevisionUi')
      ->willReturn(FALSE);

    $this->entity->expects($this->exactly(2))
      ->method('getEntityType')
      ->willReturn($entity_type);

    $form = $this->form->form([], $form_state);

    $this->assertIsArray($form);
    $this->assertArrayHasKey('crm_contact', $form);
    $this->assertArrayHasKey('user', $form);
    $this->assertTrue($form['crm_contact']['widget']['#disabled']);
    $this->assertTrue($form['user']['widget']['#disabled']);
  }

  /**
   * Tests the form method with existing entity.
   *
   * @covers ::form
   */
  public function testFormWithExistingEntity(): void {
    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->expects($this->once())
      ->method('get')
      ->with('form_display')
      ->willReturn($this->createMock(EntityFormDisplayInterface::class));

    $this->request->query->set('crm_contact', NULL);
    $this->request->query->set('user', NULL);

    $this->entity->expects($this->never())
      ->method('setContactId');
    $this->entity->expects($this->never())
      ->method('setUserId');

    $this->entity->expects($this->once())
      ->method('isNew')
      ->willReturn(FALSE);

    $entity_type = $this->createMock(EntityTypeInterface::class);
    $entity_type->expects($this->exactly(2))
      ->method('showRevisionUi')
      ->willReturn(FALSE);

    $this->entity->expects($this->exactly(2))
      ->method('getEntityType')
      ->willReturn($entity_type);

    $form = $this->form->form([], $form_state);

    $this->assertIsArray($form);
    $this->assertArrayHasKey('user', $form);
    $this->assertTrue($form['user']['widget']['#disabled']);
  }

  /**
   * Tests the save method with new entity.
   *
   * @covers ::save
   */
  public function testSaveNewEntity(): void {
    $form_state = $this->createMock(FormStateInterface::class);

    $this->entity->expects($this->once())
      ->method('save')
      ->willReturn(self::SAVED_NEW);

    $link = $this->createMock('Drupal\Core\Link');
    $link->expects($this->once())
      ->method('toString')
      ->willReturn('<a href="/view">View</a>');

    $this->entity->expects($this->once())
      ->method('toLink')
      ->with($this->callback(function ($argument) {
        return $argument instanceof TranslatableMarkup && (string) $argument === 'View';
      }))
      ->willReturn($link);

    $this->messenger->expects($this->once())
      ->method('addMessage')
      ->with($this->callback(function ($argument) {
        return $argument instanceof TranslatableMarkup && (string) $argument === 'New user contact mapping has been created.';
      }));

    $this->logger->expects($this->once())
      ->method('notice')
      ->with('Created new user contact mapping', ['link' => '<a href="/view">View</a>']);

    $form_state->expects($this->once())
      ->method('setRedirect')
      ->with('entity.crm_user_contact.collection');

    $result = $this->form->save([], $form_state);
    $this->assertEquals(self::SAVED_NEW, $result);
  }

  /**
   * Tests the save method with existing entity.
   *
   * @covers ::save
   */
  public function testSaveExistingEntity(): void {
    $form_state = $this->createMock(FormStateInterface::class);

    $this->entity->expects($this->once())
      ->method('save')
      ->willReturn(self::SAVED_UPDATED);

    $link = $this->createMock('Drupal\Core\Link');
    $link->expects($this->once())
      ->method('toString')
      ->willReturn('<a href="/view">View</a>');

    $this->entity->expects($this->once())
      ->method('toLink')
      ->with($this->callback(function ($argument) {
        return $argument instanceof TranslatableMarkup && (string) $argument === 'View';
      }))
      ->willReturn($link);

    $this->messenger->expects($this->once())
      ->method('addMessage')
      ->with($this->callback(function ($argument) {
        return $argument instanceof TranslatableMarkup && (string) $argument === 'The user contact mapping has been updated.';
      }));

    $this->logger->expects($this->once())
      ->method('notice')
      ->with('User contact mapping updated', ['link' => '<a href="/view">View</a>']);

    $form_state->expects($this->once())
      ->method('setRedirect')
      ->with('entity.crm_user_contact.collection');

    $result = $this->form->save([], $form_state);
    $this->assertEquals(self::SAVED_UPDATED, $result);
  }

}
