<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Unit\Plugin\Validation\Constraint;

use Drupal\crm\Plugin\Validation\Constraint\RelationshipContactsConstraint;
use Drupal\crm\Plugin\Validation\Constraint\RelationshipContactsConstraintValidator;
use Drupal\Tests\UnitTestCase;
use Symfony\Component\Validator\Context\ExecutionContextInterface;

/**
 * Tests the RelationshipContactsConstraintValidator.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Plugin\Validation\Constraint\RelationshipContactsConstraintValidator
 */
class RelationshipContactsConstraintValidatorTest extends UnitTestCase {

  /**
   * The validator under test.
   *
   * @var \Drupal\crm\Plugin\Validation\Constraint\RelationshipContactsConstraintValidator
   */
  protected $validator;

  /**
   * The constraint.
   *
   * @var \Drupal\crm\Plugin\Validation\Constraint\RelationshipContactsConstraint
   */
  protected $constraint;

  /**
   * The execution context.
   *
   * @var \Symfony\Component\Validator\Context\ExecutionContextInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $context;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->validator = new RelationshipContactsConstraintValidator();
    $this->constraint = new RelationshipContactsConstraint();
    $this->context = $this->createMock(ExecutionContextInterface::class);
    $this->validator->initialize($this->context);
  }

  /**
   * Tests validation passes when contacts are different and have correct types.
   *
   * @covers ::validate
   */
  public function testValidatePassesWithDifferentContactsAndCorrectTypes(): void {
    $entity = $this->createRelationshipEntity(1, 2, 'person', 'organization', 'person', 'organization');

    $this->context->expects($this->never())
      ->method('addViolation');

    $this->context->expects($this->never())
      ->method('buildViolation');

    $this->validator->validate($entity, $this->constraint);
  }

  /**
   * Tests validation fails when both contacts are the same.
   *
   * @covers ::validate
   */
  public function testValidateFailsWhenContactsAreSame(): void {
    $entity = $this->createRelationshipEntity(1, 1, 'person', 'person', 'person', 'person');

    $this->context->expects($this->once())
      ->method('addViolation')
      ->with($this->constraint->differentMessage);

    $this->validator->validate($entity, $this->constraint);
  }

  /**
   * Tests validation fails when contact A has wrong type.
   *
   * @covers ::validate
   */
  public function testValidateFailsWhenContactOneHasWrongType(): void {
    $entity = $this->createRelationshipEntity(1, 2, 'organization', 'organization', 'person', 'organization');

    $violation_builder = $this->createMock('Symfony\Component\Validator\Violation\ConstraintViolationBuilderInterface');
    $violation_builder->expects($this->once())
      ->method('atPath')
      ->with('contact_a')
      ->willReturnSelf();
    $violation_builder->expects($this->once())
      ->method('setParameter')
      ->with('@type', 'person')
      ->willReturnSelf();
    $violation_builder->expects($this->once())
      ->method('addViolation');

    $this->context->expects($this->once())
      ->method('buildViolation')
      ->with($this->constraint->wrongTypeMessage)
      ->willReturn($violation_builder);

    $this->validator->validate($entity, $this->constraint);
  }

  /**
   * Tests validation fails when contact B has wrong type.
   *
   * @covers ::validate
   */
  public function testValidateFailsWhenContactTwoHasWrongType(): void {
    $entity = $this->createRelationshipEntity(1, 2, 'person', 'person', 'person', 'organization');

    $violation_builder = $this->createMock('Symfony\Component\Validator\Violation\ConstraintViolationBuilderInterface');
    $violation_builder->expects($this->once())
      ->method('atPath')
      ->with('contact_b')
      ->willReturnSelf();
    $violation_builder->expects($this->once())
      ->method('setParameter')
      ->with('@type', 'organization')
      ->willReturnSelf();
    $violation_builder->expects($this->once())
      ->method('addViolation');

    $this->context->expects($this->once())
      ->method('buildViolation')
      ->with($this->constraint->wrongTypeMessage)
      ->willReturn($violation_builder);

    $this->validator->validate($entity, $this->constraint);
  }

  /**
   * Tests validation fails when both contacts have wrong types.
   *
   * @covers ::validate
   */
  public function testValidateFailsWhenBothContactsHaveWrongTypes(): void {
    $entity = $this->createRelationshipEntity(1, 2, 'organization', 'person', 'person', 'organization');

    $violation_builder_a = $this->createMock('Symfony\Component\Validator\Violation\ConstraintViolationBuilderInterface');
    $violation_builder_a->expects($this->once())
      ->method('atPath')
      ->with('contact_a')
      ->willReturnSelf();
    $violation_builder_a->expects($this->once())
      ->method('setParameter')
      ->with('@type', 'person')
      ->willReturnSelf();
    $violation_builder_a->expects($this->once())
      ->method('addViolation');

    $violation_builder_b = $this->createMock('Symfony\Component\Validator\Violation\ConstraintViolationBuilderInterface');
    $violation_builder_b->expects($this->once())
      ->method('atPath')
      ->with('contact_b')
      ->willReturnSelf();
    $violation_builder_b->expects($this->once())
      ->method('setParameter')
      ->with('@type', 'organization')
      ->willReturnSelf();
    $violation_builder_b->expects($this->once())
      ->method('addViolation');

    $this->context->expects($this->exactly(2))
      ->method('buildViolation')
      ->with($this->constraint->wrongTypeMessage)
      ->willReturnOnConsecutiveCalls($violation_builder_a, $violation_builder_b);

    $this->validator->validate($entity, $this->constraint);
  }

  /**
   * Tests validation when contact A is null.
   *
   * @covers ::validate
   */
  public function testValidateWithNullContactA(): void {
    $entity = $this->createRelationshipEntity(NULL, 2, NULL, 'organization', 'person', 'organization');

    $violation_builder = $this->createMock('Symfony\Component\Validator\Violation\ConstraintViolationBuilderInterface');
    $violation_builder->expects($this->once())
      ->method('atPath')
      ->with('contact_a')
      ->willReturnSelf();
    $violation_builder->expects($this->once())
      ->method('setParameter')
      ->with('@type', 'person')
      ->willReturnSelf();
    $violation_builder->expects($this->once())
      ->method('addViolation');

    $this->context->expects($this->once())
      ->method('buildViolation')
      ->with($this->constraint->wrongTypeMessage)
      ->willReturn($violation_builder);

    $this->validator->validate($entity, $this->constraint);
  }

  /**
   * Tests validation when contact B is null.
   *
   * @covers ::validate
   */
  public function testValidateWithNullContactB(): void {
    $entity = $this->createRelationshipEntity(1, NULL, 'person', NULL, 'person', 'organization');

    $violation_builder = $this->createMock('Symfony\Component\Validator\Violation\ConstraintViolationBuilderInterface');
    $violation_builder->expects($this->once())
      ->method('atPath')
      ->with('contact_b')
      ->willReturnSelf();
    $violation_builder->expects($this->once())
      ->method('setParameter')
      ->with('@type', 'organization')
      ->willReturnSelf();
    $violation_builder->expects($this->once())
      ->method('addViolation');

    $this->context->expects($this->once())
      ->method('buildViolation')
      ->with($this->constraint->wrongTypeMessage)
      ->willReturn($violation_builder);

    $this->validator->validate($entity, $this->constraint);
  }

  /**
   * Tests validation when both contacts are null.
   *
   * @covers ::validate
   */
  public function testValidateWithBothContactsNull(): void {
    $entity = $this->createRelationshipEntity(NULL, NULL, NULL, NULL, 'person', 'organization');

    $violation_builder_a = $this->createMock('Symfony\Component\Validator\Violation\ConstraintViolationBuilderInterface');
    $violation_builder_a->expects($this->once())
      ->method('atPath')
      ->with('contact_a')
      ->willReturnSelf();
    $violation_builder_a->expects($this->once())
      ->method('setParameter')
      ->with('@type', 'person')
      ->willReturnSelf();
    $violation_builder_a->expects($this->once())
      ->method('addViolation');

    $violation_builder_b = $this->createMock('Symfony\Component\Validator\Violation\ConstraintViolationBuilderInterface');
    $violation_builder_b->expects($this->once())
      ->method('atPath')
      ->with('contact_b')
      ->willReturnSelf();
    $violation_builder_b->expects($this->once())
      ->method('setParameter')
      ->with('@type', 'organization')
      ->willReturnSelf();
    $violation_builder_b->expects($this->once())
      ->method('addViolation');

    // When both contacts are null, they're both NULL === NULL, so it will add
    // the differentMessage violation as well.
    $this->context->expects($this->once())
      ->method('addViolation')
      ->with($this->constraint->differentMessage);

    $this->context->expects($this->exactly(2))
      ->method('buildViolation')
      ->with($this->constraint->wrongTypeMessage)
      ->willReturnOnConsecutiveCalls($violation_builder_a, $violation_builder_b);

    $this->validator->validate($entity, $this->constraint);
  }

  /**
   * Tests validation when both contacts are the same but also have wrong types.
   *
   * @covers ::validate
   */
  public function testValidateWithSameContactsAndWrongTypes(): void {
    $entity = $this->createRelationshipEntity(1, 1, 'organization', 'organization', 'person', 'household');

    $violation_builder_a = $this->createMock('Symfony\Component\Validator\Violation\ConstraintViolationBuilderInterface');
    $violation_builder_a->expects($this->once())
      ->method('atPath')
      ->with('contact_a')
      ->willReturnSelf();
    $violation_builder_a->expects($this->once())
      ->method('setParameter')
      ->with('@type', 'person')
      ->willReturnSelf();
    $violation_builder_a->expects($this->once())
      ->method('addViolation');

    $violation_builder_b = $this->createMock('Symfony\Component\Validator\Violation\ConstraintViolationBuilderInterface');
    $violation_builder_b->expects($this->once())
      ->method('atPath')
      ->with('contact_b')
      ->willReturnSelf();
    $violation_builder_b->expects($this->once())
      ->method('setParameter')
      ->with('@type', 'household')
      ->willReturnSelf();
    $violation_builder_b->expects($this->once())
      ->method('addViolation');

    $this->context->expects($this->once())
      ->method('addViolation')
      ->with($this->constraint->differentMessage);

    $this->context->expects($this->exactly(2))
      ->method('buildViolation')
      ->with($this->constraint->wrongTypeMessage)
      ->willReturnOnConsecutiveCalls($violation_builder_a, $violation_builder_b);

    $this->validator->validate($entity, $this->constraint);
  }

  /**
   * Creates a mock relationship entity with the specified parameters.
   *
   * @param int|null $contact_a_id
   *   The ID of contact A, or NULL.
   * @param int|null $contact_b_id
   *   The ID of contact B, or NULL.
   * @param string|null $contact_a_type
   *   The bundle type of contact A, or NULL.
   * @param string|null $contact_b_type
   *   The bundle type of contact B, or NULL.
   * @param string $expected_contact_a_type
   *   The expected bundle type for contact A.
   * @param string $expected_contact_b_type
   *   The expected bundle type for contact B.
   *
   * @return \Drupal\Core\Entity\EntityInterface|\PHPUnit\Framework\MockObject\MockObject
   *   The mock entity.
   */
  protected function createRelationshipEntity(
    ?int $contact_a_id,
    ?int $contact_b_id,
    ?string $contact_a_type,
    ?string $contact_b_type,
    string $expected_contact_a_type,
    string $expected_contact_b_type,
  ) {
    // Create mock contacts.
    $contact_a = NULL;
    if ($contact_a_id !== NULL) {
      $contact_a = $this->createMock('Drupal\crm\Entity\Contact');
      $contact_a->expects($this->any())
        ->method('id')
        ->willReturn((string) $contact_a_id);
      $contact_a->expects($this->any())
        ->method('bundle')
        ->willReturn($contact_a_type);
    }

    $contact_b = NULL;
    if ($contact_b_id !== NULL) {
      $contact_b = $this->createMock('Drupal\crm\Entity\Contact');
      $contact_b->expects($this->any())
        ->method('id')
        ->willReturn((string) $contact_b_id);
      $contact_b->expects($this->any())
        ->method('bundle')
        ->willReturn($contact_b_type);
    }

    // Create mock contacts field that returns an array with contacts.
    // The validator accesses [0] and [1] directly, so we need these indices.
    $referenced_entities = [$contact_a, $contact_b];

    $contacts_field = $this->createMock('Drupal\Core\Field\EntityReferenceFieldItemListInterface');
    $contacts_field->expects($this->any())
      ->method('referencedEntities')
      ->willReturn($referenced_entities);

    // Create mock relationship type.
    $relationship_type = $this->createMock('Drupal\crm\Entity\RelationshipType');
    $relationship_type->expects($this->any())
      ->method('get')
      ->willReturnCallback(function ($key) use ($expected_contact_a_type, $expected_contact_b_type) {
        if ($key === 'contact_type_a') {
          return $expected_contact_a_type;
        }
        if ($key === 'contact_type_b') {
          return $expected_contact_b_type;
        }
        return NULL;
      });

    // Create mock bundle field with entity property.
    // Using stdClass to provide a simple object with the entity property.
    $bundle_field = new \stdClass();
    $bundle_field->entity = $relationship_type;

    // Create mock entity.
    $entity = $this->createMock('Drupal\crm\Entity\Relationship');
    $entity->expects($this->any())
      ->method('get')
      ->willReturnCallback(function ($field_name) use ($contacts_field, $bundle_field) {
        if ($field_name === 'contacts') {
          return $contacts_field;
        }
        if ($field_name === 'bundle') {
          return $bundle_field;
        }
        return NULL;
      });

    return $entity;
  }

}
