<?php

declare(strict_types=1);

namespace Drupal\crm\Controller;

use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Returns responses for Crm routes.
 */
class RelationshipController extends ControllerBase {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * The controller constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new self(
      $container->get('entity_type.manager')
    );
  }

  /**
   * Builds the response.
   */
  public function build($crm_contact) {

    $build['active'] = [
      '#type' => 'table',
      '#header' => $this->buildHeader(),
      '#title' => $this->t('Active Relationships'),
      '#rows' => $this->getActiveRelationships($crm_contact),
      '#empty' => $this->t('There are no active relationships.'),
      '#cache' => [],
    ];

    $build['inactive'] = [
      '#type' => 'table',
      '#header' => $this->buildHeader(),
      '#title' => $this->t('Inactive Relationships'),
      '#rows' => $this->getInactiveRelationships($crm_contact),
      '#empty' => $this->t('There are no inactive relationships.'),
      '#cache' => [],
    ];

    $build['#cache']['max-age'] = 0;

    return $build;
  }

  /**
   * Builds the header.
   *
   * @return array
   *   An array of header cells.
   */
  protected function buildHeader() {
    $header['type'] = $this->t('Relationship');
    $header['contact'] = $this->t('Contact');
    $header['start_date'] = $this->t('Start Date');
    $header['end_date'] = $this->t('End Date');
    $header['address'] = $this->t('Address');
    $header['email'] = $this->t('Email');
    $header['phone'] = $this->t('Phone');

    $header['operations'] = $this->t('Operations');
    return $header;
  }

  /**
   * Gets the active relationships.
   *
   * @param int $crm_contact
   *   The CRM contact ID.
   *
   * @return array
   *   An array of active relationships.
   */
  protected function getActiveRelationships($crm_contact) {

    $storage = $this->entityTypeManager->getStorage('crm_relationship');
    $query = $storage->getQuery();
    $query->condition('status', 1)
      ->sort('start_date', 'DESC');
    $or = $query->orConditionGroup()
      ->condition('contacts', $crm_contact);
    $query->condition($or);
    $result = $query->accessCheck(FALSE)->execute();
    $relationships = $storage->loadMultiple($result);

    return $this->doRows($relationships, $crm_contact);
  }

  /**
   * Gets the inactive relationships.
   *
   * @param int $crm_contact
   *   The CRM contact ID.
   *
   * @return array
   *   An array of inactive relationships.
   */
  protected function getInactiveRelationships($crm_contact) {

    $storage = $this->entityTypeManager->getStorage('crm_relationship');
    $query = $storage->getQuery();
    $query->condition('status', 0)
      ->sort('end_date', 'DESC');
    $or = $query->orConditionGroup()
      ->condition('contacts', $crm_contact);
    $query->condition($or);
    $result = $query->accessCheck(FALSE)->execute();
    $relationships = $storage->loadMultiple($result);

    return $this->doRows($relationships, $crm_contact);
  }

  /**
   * Builds the rows.
   *
   * @param array $relationships
   *   An array of relationships.
   * @param int $crm_contact
   *   The CRM contact ID.
   *
   * @return array
   *   An array of rows.
   */
  protected function doRows($relationships, $crm_contact) {
    $rows = [];

    foreach ($relationships as $relationship) {
      $contacts = $relationship->get('contacts')->referencedEntities();
      $is_a = $contacts[0]->id() == $crm_contact;
      $contact = $is_a ? $contacts[1] : $contacts[0];
      $contact_label = $contact->toLink()->toString();
      $relationship_label = $is_a ? $relationship->bundle->entity->get('label_b') : $relationship->bundle->entity->get('label_a');
      $rows[] = [
        'type' => $relationship_label,
        'contact' => $contact_label,
        'start_date' => $relationship->get('start_date')->value,
        'end_date' => $relationship->get('end_date')->value,
        'address' => $contact->get('addresses')->primary()?->entity->label(),
        'email' => $contact->get('emails')->primary()?->entity->label(),
        'phone' => $contact->get('telephones')->primary()?->entity->label(),
        'operations' => [
          'data' => [
            '#type' => 'operations',
            '#links' => [
              'edit' => [
                'title' => $this->t('Edit'),
                'url' => $relationship->toUrl('edit-form'),
              ],
              'delete' => [
                'title' => $this->t('Delete'),
                'url' => $relationship->toUrl('delete-form'),
              ],
            ],
          ],
        ],
      ];
    }
    return $rows;
  }

}
