<?php

namespace Drupal\crm;

use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\BundlePermissionHandlerTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides dynamic permissions for each contact type.
 */
class CrmContactTypePermissions implements ContainerInjectionInterface {
  use BundlePermissionHandlerTrait;
  use StringTranslationTrait;

  /**
   * The entity type manager service.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * CrmContactTypePermissions constructor.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager service.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager) {
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static($container->get('entity_type.manager'));
  }

  /**
   * Returns an array of contact type permissions.
   *
   * @return array
   *   The contact type permissions.
   */
  public function contactTypePermissions() {
    $contact_types = $this->entityTypeManager->getStorage('crm_contact_type')->loadMultiple();
    return $this->generatePermissions($contact_types, [$this, 'buildPermissions']);
  }

  /**
   * Returns a list of permissions for a given contact type.
   *
   * @param \Drupal\crm\CrmContactTypeInterface $type
   *   The contact type.
   *
   * @return array
   *   An associative array of permission names and descriptions.
   */
  protected function buildPermissions($type) {
    $type_id = $type->id();
    $type_params = ['%type_name' => $type->label()];

    return [
      "create $type_id crm_contact" => [
        'title' => $this->t('%type_name: Create new contact', $type_params),
      ],
      "edit any $type_id crm_contact" => [
        'title' => $this->t('%type_name: Edit any contact', $type_params),
      ],
      "delete any $type_id crm_contact" => [
        'title' => $this->t('%type_name: Delete any contact', $type_params),
      ],
      "view any $type_id crm_contact" => [
        'title' => $this->t('%type_name: View any contact', $type_params),
      ],
      "view any $type_id crm_contact label" => [
        'title' => $this->t('%type_name: View any contact', $type_params),
      ],
    ];
  }

}
