<?php

declare(strict_types=1);

namespace Drupal\crm\Entity;

use Drupal\Core\Entity\Attribute\ContentEntityType;
use Drupal\Core\Entity\ContentEntityDeleteForm;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityPublishedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\RevisionableContentEntityBase;
use Drupal\Core\Entity\RevisionLogEntityTrait;
use Drupal\Core\Entity\Routing\AdminHtmlRouteProvider;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\crm\CrmRelationshipInterface;
use Drupal\crm\Field\AgeFieldItemList;
use Drupal\crm\Field\RelationshipContactsItemList;
use Drupal\crm\Form\RelationshipForm;
use Drupal\crm\RelationshipAccessControlHandler;
use Drupal\crm\RelationshipListBuilder;
use Drupal\views\EntityViewsData;

/**
 * CRM relationship entity class.
 *
 * Two contacts can have a relationship.
 */
#[ContentEntityType(
  id: 'crm_relationship',
  label: new TranslatableMarkup('CRM Relationship'),
  label_collection: new TranslatableMarkup('CRM Relationships'),
  label_singular: new TranslatableMarkup('crm relationship'),
  label_plural: new TranslatableMarkup('crm relationships'),
  label_count: [
    'singular' => '@count crm relationship',
    'plural' => '@count crm relationships',
  ],
  bundle_label: new TranslatableMarkup('CRM Relationship type'),
  handlers: [
    'list_builder' => RelationshipListBuilder::class,
    'views_data' => EntityViewsData::class,
    'form' => [
      'default' => RelationshipForm::class,
      'delete' => ContentEntityDeleteForm::class,
    ],
    'access' => RelationshipAccessControlHandler::class,
    'route_provider' => [
      'html' => AdminHtmlRouteProvider::class,
    ],
  ],
  base_table: 'crm_relationship',
  revision_table: 'crm_relationship_revision',
  show_revision_ui: TRUE,
  admin_permission: 'administer crm',
  entity_keys: [
    'id' => 'id',
    'revision' => 'revision_id',
    'bundle' => 'bundle',
    'uuid' => 'uuid',
    'status' => 'status',
    'published' => 'status',
  ],
  revision_metadata_keys: [
    'revision_user' => 'revision_uid',
    'revision_created' => 'revision_timestamp',
    'revision_log_message' => 'revision_log',
  ],
  bundle_entity_type: 'crm_relationship_type',
  field_ui_base_route: 'entity.crm_relationship_type.edit_form',
  translatable: FALSE,
  constraints: [
    'RelationshipContacts' => [],
  ],
  links: [
    'collection' => '/admin/content/crm/relationship',
    'add-form' => '/crm/relationship/add/{crm_relationship_type}',
    'add-page' => '/crm/relationship/add',
    'canonical' => '/crm/relationship/{crm_relationship}',
    'edit-form' => '/crm/relationship/{crm_relationship}/edit',
    'delete-form' => '/crm/relationship/{crm_relationship}/delete',
  ],
)]
class Relationship extends RevisionableContentEntityBase implements CrmRelationshipInterface {

  use EntityChangedTrait;
  use EntityPublishedTrait;
  use RevisionLogEntityTrait;

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type) {

    $fields = parent::baseFieldDefinitions($entity_type);
    // Add the published field.
    $fields += static::publishedBaseFieldDefinitions($entity_type);
    $fields['contacts'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel('Contacts')
      ->setSetting('target_type', 'crm_contact')
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', FALSE)
      ->setCardinality(2)
      ->setRequired(TRUE)
      ->setRevisionable(TRUE);

    $fields['contact_a'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel('Contact A')
      ->setDescription(t('The first contact in the relationship.'))
      ->setComputed(TRUE)
      ->setClass(RelationshipContactsItemList::class)
      ->setSetting('target_type', 'crm_contact')
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'entity_reference_label',
        'label' => 'above',
        'weight' => 0,
      ])
      ->setRequired(TRUE);

    $fields['contact_b'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel('Contact B')
      ->setDescription(t('The second contact in the relationship.'))
      ->setComputed(TRUE)
      ->setClass(RelationshipContactsItemList::class)
      ->setSetting('target_type', 'crm_contact')
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'entity_reference_autocomplete',
        'weight' => 0,
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'entity_reference_label',
        'label' => 'above',
        'weight' => 0,
      ])
      ->setRequired(TRUE);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setRevisionable(TRUE)
      ->setLabel(t('Status'))
      ->setDefaultValue(TRUE)
      ->setSetting('on_label', 'Status')
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'settings' => [
          'display_label' => FALSE,
        ],
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'boolean',
        'label' => 'above',
        'weight' => 0,
        'settings' => [
          'format' => 'enabled-disabled',
        ],
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['start_date'] = BaseFieldDefinition::create('datetime')
      ->setLabel(t('Start Date'))
      ->setDescription(t('When the relationship started.'))
      ->setRevisionable(TRUE)
      ->setSettings([
        'datetime_type' => 'date',
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'datetime_default',
        'settings' => [
          'format_type' => 'medium',
        ],
        'weight' => -9,
      ])
      ->setDisplayOptions('form', [
        'type' => 'datetime_default',
        'weight' => -9,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRevisionable(TRUE);

    $fields['end_date'] = BaseFieldDefinition::create('datetime')
      ->setLabel(t('End Date'))
      ->setDescription(t('When the relationship ends.'))
      ->setRevisionable(TRUE)
      ->setSettings([
        'datetime_type' => 'date',
      ])
      ->setDefaultValue('')
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'datetime_default',
        'settings' => [
          'format_type' => 'medium',
        ],
        'weight' => -9,
      ])
      ->setDisplayOptions('form', [
        'type' => 'datetime_default',
        'weight' => -9,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayConfigurable('view', TRUE)
      ->setRevisionable(TRUE);

    $fields['age'] = BaseFieldDefinition::create('integer')
      ->setLabel(t('Age'))
      ->setDescription(t('The age of the relationship.'))
      ->setComputed(TRUE)
      ->setClass(AgeFieldItemList::class)
      ->setDisplayConfigurable('form', FALSE)
      ->setDisplayConfigurable('view', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'age_formatter',
        'settings' => [
          'granularity' => 'years',
        ],
        'weight' => -9,
      ]);

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created on'))
      ->setDescription(t('The time that the crm relationship was created.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('view', TRUE)
      ->setRevisionable(TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the crm relationship was last edited.'))
      ->setRevisionable(TRUE);

    return $fields;
  }

  /**
   * {@inheritdoc}
   */
  public function label() {
    // Get the entity bundle entity.
    $bundle = $this->get('bundle')->entity;
    $label = $bundle->get('label');
    $contact_a = $this->get('contact_a')->entity->label();
    $contact_b = $this->get('contact_b')->entity->label();

    return "$label ($contact_a<=>$contact_b)";
  }

}
