<?php

namespace Drupal\crm\Form;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form controller for the crm relationship entity edit forms.
 */
class RelationshipForm extends ContentEntityForm {

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * Constructs a RelationshipForm object.
   *
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface|null $entity_type_bundle_info
   *   The entity type bundle service.
   */
  public function __construct(
    EntityRepositoryInterface $entity_repository,
    TimeInterface $time,
    DateFormatterInterface $date_formatter,
    AccountInterface $current_user,
    ?EntityTypeBundleInfoInterface $entity_type_bundle_info = NULL,
  ) {

    parent::__construct($entity_repository, $entity_type_bundle_info, $time);
    $this->dateFormatter = $date_formatter;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  final public static function create(ContainerInterface $container) {
    return new self(
      $container->get('entity.repository'),
      $container->get('datetime.time'),
      $container->get('date.formatter'),
      $container->get('current_user'),
      $container->get('entity_type.bundle.info'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);
    $relationship = $this->getEntity();
    $bundle = $relationship->bundle->entity;

    $form['contact_a']['widget'][0]['target_id']['#selection_settings']['target_bundles'] = [
      $bundle->get('contact_type_a') => $bundle->get('contact_type_a'),
    ];
    $form['contact_a']['widget'][0]['target_id']['#title'] = $bundle->get('label_a');

    $form['contact_b']['widget'][0]['target_id']['#selection_settings']['target_bundles'] = [
      $bundle->get('contact_type_b') => $bundle->get('contact_type_b'),
    ];
    $form['contact_b']['widget'][0]['target_id']['#title'] = $bundle->get('label_b');

    if ($this->getOperation() == 'edit') {
      $form['contact_a']['widget']['#disabled'] = TRUE;
      $form['contact_b']['widget']['#disabled'] = TRUE;
    }

    $form['advanced']['#attributes']['class'][] = 'entity-meta';

    $form['meta'] = [
      '#type' => 'details',
      '#group' => 'advanced',
      '#weight' => -10,
      '#title' => $this->t('Status'),
      '#attributes' => ['class' => ['entity-meta__header']],
      '#tree' => TRUE,
      '#access' => $this->currentUser->hasPermission('administer crm'),
    ];
    $form['meta']['published'] = [
      '#type' => 'item',
      '#markup' => $relationship->isPublished() ? $this->t('Active') : $this->t('Inactive'),
      '#access' => !$relationship->isNew(),
      '#wrapper_attributes' => ['class' => ['entity-meta__title']],
    ];
    $form['meta']['changed'] = [
      '#type' => 'item',
      '#title' => $this->t('Last saved'),
      '#markup' => !$relationship->isNew() ? $this->dateFormatter->format($relationship->getChangedTime(), 'short') : $this->t('Not saved yet'),
      '#wrapper_attributes' => ['class' => ['entity-meta__last-saved']],
    ];

    $form['meta']['status'] = &$form['status'];
    $form['meta']['status']['#weight'] = 100;
    unset($form['status']);

    if (isset($form['uid'])) {
      unset($form['uid']);
    }

    if (isset($form['created'])) {
      $form['created']['#weight'] = 200;
      $form['meta']['created'] = &$form['created'];
      unset($form['created']);
    }

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $result = parent::save($form, $form_state);

    $relationship = $this->getEntity();

    $label = $relationship->label() ?? 'No label';
    $message_arguments = ['%label' => $label];
    $logger_arguments = [
      '%label' => $label,
      'link' => $relationship->toLink($this->t('View'))->toString(),
    ];

    switch ($result) {
      case SAVED_NEW:
        $this->messenger()->addStatus($this->t('New crm relationship %label has been created.', $message_arguments));
        $this->logger('crm')->notice('Created new crm relationship %label', $logger_arguments);
        break;

      case SAVED_UPDATED:
        $this->messenger()->addStatus($this->t('The crm relationship %label has been updated.', $message_arguments));
        $this->logger('crm')->notice('Updated crm relationship %label.', $logger_arguments);
        break;
    }

    $form_state->setRedirect('entity.crm_relationship.canonical', ['crm_relationship' => $relationship->id()]);

    return $result;
  }

}
