<?php

namespace Drupal\crm\Plugin\Menu\LocalTask;

use Drupal\Core\Menu\LocalTaskDefault;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\comment\CommentStorageInterface;
use Drupal\crm\CrmContactInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;

/**
 * Provides a local task that shows the amount of comments.
 */
class Comment extends LocalTaskDefault implements ContainerFactoryPluginInterface {
  use StringTranslationTrait;

  /**
   * The comment storage service.
   *
   * @var \Drupal\comment\CommentStorageInterface
   */
  protected $storage;

  /**
   * The route match service.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * Construct the UnapprovedComments object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param array $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\comment\CommentStorageInterface $comment_storage
   *   The comment storage service.
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    array $plugin_definition,
    CommentStorageInterface $comment_storage,
    RouteMatchInterface $route_match,
  ) {

    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->storage = $comment_storage;
    $this->routeMatch = $route_match;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new self(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')->getStorage('comment'),
      $container->get('current_route_match')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getTitle(?Request $request = NULL) {
    $contact_id = $this->routeMatch->getParameter('crm_contact');
    if ($contact_id instanceof CrmContactInterface) {
      $contact_id = $contact_id->id();
    }
    $count = $this->storage->getQuery()
      ->condition('entity_type', 'crm_contact')
      ->condition('entity_id', $contact_id)
      ->condition('status', 1)
      ->condition('field_name', 'comment')
      ->accessCheck(TRUE)
      ->count()
      ->execute();

    return $this->formatPlural($count, 'Comment (1)', 'Comments (@count)');
  }

  /**
   * {@inheritdoc}
   */
  public function getCacheTags() {

    $contact_id = $this->routeMatch->getParameter('crm_contact');
    if ($contact_id instanceof CrmContactInterface) {
      $contact_id = $contact_id->id();
    }

    return ['crm_contact:' . $contact_id];
  }

}
