<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Functional\Form;

use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\UserContact;
use Drupal\Tests\BrowserTestBase;

/**
 * Functional tests for the UserContactForm.
 *
 * @group crm
 * @covers \Drupal\crm\Form\UserContactForm
 */
class UserContactFormTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'comment',
    'datetime',
    'name',
    'telephone',
    'address',
  ];

  /**
   * A user with CRM administration permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * A normal user without CRM administration permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $normalUser;

  /**
   * Test contact.
   *
   * @var \Drupal\crm\Entity\Contact
   */
  protected $testContact;

  /**
   * Another test contact.
   *
   * @var \Drupal\crm\Entity\Contact
   */
  protected $anotherContact;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->adminUser = $this->createUser([
      'administer crm',
      'view any crm_contact label',
      'view any crm_contact',
      'view any person crm_contact',
      'edit any crm_contact',
    ]);

    $this->normalUser = $this->createUser([
      'view any crm_contact',
    ]);

    // Create test contacts.
    $this->testContact = Contact::create([
      'bundle' => 'person',
      'full_name' => [
        'given' => 'Test',
        'family' => 'Contact',
      ],
    ]);
    $this->testContact->save();

    $this->anotherContact = Contact::create([
      'bundle' => 'person',
      'full_name' => [
        'given' => 'Another',
        'family' => 'Contact',
      ],
    ]);
    $this->anotherContact->save();
  }

  /**
   * Tests access to the user contact forms.
   */
  public function testUserContactFormAccess(): void {
    // Test that anonymous users cannot access the form.
    $this->drupalGet('admin/config/crm/user/add');
    $this->assertSession()->statusCodeEquals(403);

    // Test that normal users cannot access the form.
    $this->drupalLogin($this->normalUser);
    $this->drupalGet('admin/config/crm/user/add');
    $this->assertSession()->statusCodeEquals(403);

    // Test that admin users can access the form.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/add');
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests creating a new user contact mapping.
   */
  public function testCreateUserContactMapping(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/add');

    // Verify form elements are present.
    $this->assertSession()->fieldExists('user[0][target_id]');
    $this->assertSession()->fieldExists('crm_contact[0][target_id]');
    $this->assertSession()->buttonExists('Save');

    // Create a new user for testing.
    $test_user = $this->createUser([], 'mapped_user');

    // Submit form with valid data.
    $edit = [
      'user[0][target_id]' => $test_user->getAccountName() . ' (' . $test_user->id() . ')',
      'crm_contact[0][target_id]' => $this->testContact->label() . ' (' . $this->testContact->id() . ')',
    ];
    $this->submitForm($edit, 'Save');

    // Verify success message.
    $this->assertSession()->pageTextContains('New user contact mapping has been created.');

    // Verify entity was created.
    $query = \Drupal::entityQuery('crm_user_contact')
      ->condition('user', $test_user->id())
      ->condition('crm_contact', $this->testContact->id())
      ->accessCheck(FALSE);
    $results = $query->execute();

    $this->assertCount(1, $results, 'User contact mapping was created.');

    // Verify redirect to collection page.
    $this->assertSession()->addressEquals('admin/config/crm/user/list');
  }

  /**
   * Tests creating a user contact mapping with query parameters.
   */
  public function testCreateUserContactMappingWithQueryParameters(): void {
    $this->drupalLogin($this->adminUser);

    $test_user = $this->createUser([], 'query_user');

    // Test with crm_contact query parameter.
    $this->drupalGet('admin/config/crm/user/add', [
      'query' => ['crm_contact' => $this->testContact->id()],
    ]);

    // Verify that the contact field is disabled and pre-populated.
    $this->assertSession()->fieldDisabled('crm_contact[0][target_id]');
    $this->assertSession()->fieldValueEquals('crm_contact[0][target_id]', $this->testContact->label() . ' (' . $this->testContact->id() . ')');

    // Verify that the user field is not disabled.
    $this->assertSession()->fieldEnabled('user[0][target_id]');

    // Submit form.
    $edit = [
      'user[0][target_id]' => $test_user->getAccountName() . ' (' . $test_user->id() . ')',
    ];
    $this->submitForm($edit, 'Save');

    $this->assertSession()->pageTextContains('New user contact mapping has been created.');

    // Test with user query parameter.
    $another_test_user = $this->createUser([], 'another_query_user');
    $this->drupalGet('admin/config/crm/user/add', [
      'query' => ['user' => $another_test_user->id()],
    ]);

    // Verify that the user field is disabled and pre-populated.
    $this->assertSession()->fieldDisabled('user[0][target_id]');
    $this->assertSession()->fieldValueEquals('user[0][target_id]', $another_test_user->getAccountName() . ' (' . $another_test_user->id() . ')');

    // Verify that the contact field is not disabled.
    $this->assertSession()->fieldEnabled('crm_contact[0][target_id]');

    // Submit form.
    $edit = [
      'crm_contact[0][target_id]' => $this->anotherContact->label() . ' (' . $this->anotherContact->id() . ')',
    ];
    $this->submitForm($edit, 'Save');

    $this->assertSession()->pageTextContains('New user contact mapping has been created.');
  }

  /**
   * Tests creating a user contact mapping with both query parameters.
   */
  public function testCreateUserContactMappingWithBothQueryParameters(): void {
    $this->drupalLogin($this->adminUser);

    $test_user = $this->createUser([], 'both_query_user');

    $this->drupalGet('admin/config/crm/user/add', [
      'query' => [
        'crm_contact' => $this->testContact->id(),
        'user' => $test_user->id(),
      ],
    ]);

    // Verify that both fields are disabled and pre-populated.
    $this->assertSession()->fieldDisabled('crm_contact[0][target_id]');
    $this->assertSession()->fieldDisabled('user[0][target_id]');
    $this->assertSession()->fieldValueEquals('crm_contact[0][target_id]', $this->testContact->label() . ' (' . $this->testContact->id() . ')');
    $this->assertSession()->fieldValueEquals('user[0][target_id]', $test_user->getAccountName() . ' (' . $test_user->id() . ')');

    // Submit form.
    $this->submitForm([], 'Save');

    $this->assertSession()->pageTextContains('New user contact mapping has been created.');

    // Verify entity was created with correct values.
    $query = \Drupal::entityQuery('crm_user_contact')
      ->condition('user', $test_user->id())
      ->condition('crm_contact', $this->testContact->id())
      ->accessCheck(FALSE);
    $results = $query->execute();

    $this->assertCount(1, $results, 'User contact mapping was created with both query parameters.');
  }

  /**
   * Tests editing an existing user contact mapping.
   */
  public function testEditUserContactMapping(): void {
    // Create a user contact mapping to edit.
    $test_user = $this->createUser([], 'edit_user');
    $user_contact = UserContact::create([
      'user' => $test_user->id(),
      'crm_contact' => $this->testContact->id(),
    ]);
    $user_contact->save();

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/' . $user_contact->id() . '/edit');

    // Verify form is populated with existing values.
    $this->assertSession()->fieldValueEquals('user[0][target_id]', $test_user->getAccountName() . ' (' . $test_user->id() . ')');
    $this->assertSession()->fieldValueEquals('crm_contact[0][target_id]', $this->testContact->label() . ' (' . $this->testContact->id() . ')');

    // Verify that user field is disabled for existing entities.
    $this->assertSession()->fieldDisabled('user[0][target_id]');
    $this->assertSession()->fieldEnabled('crm_contact[0][target_id]');

    // Edit the contact mapping.
    $edit = [
      'crm_contact[0][target_id]' => $this->anotherContact->label() . ' (' . $this->anotherContact->id() . ')',
    ];
    $this->submitForm($edit, 'Save');

    // Verify update message.
    $this->assertSession()->pageTextContains('The user contact mapping has been updated.');

    // Verify entity was updated.
    $updated_entity = UserContact::load($user_contact->id());
    $this->assertEquals($this->anotherContact->id(), $updated_entity->getContactId());
    $this->assertEquals($test_user->id(), $updated_entity->getUserId());

    // Verify redirect to collection page.
    $this->assertSession()->addressEquals('admin/config/crm/user/list');
  }

  /**
   * Tests form validation.
   */
  public function testUserContactFormValidation(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/add');

    // Test submitting empty form.
    $this->submitForm([], 'Save');
    $this->assertSession()->pageTextContains('User field is required.');

    // Test submitting with only user.
    $test_user = $this->createUser([], 'validation_user');
    $edit = [
      'user[0][target_id]' => $test_user->getAccountName() . ' (' . $test_user->id() . ')',
    ];
    $this->submitForm($edit, 'Save');

    $this->assertSession()->pageTextContains('Person field is required.');

    $edit['crm_contact[0][target_id]'] = $this->testContact->label() . ' (' . $this->testContact->id() . ')';
    $this->submitForm($edit, 'Save');

    // Should succeed as contact is not required.
    $this->assertSession()->pageTextContains('New user contact mapping has been created.');
  }

  /**
   * Tests duplicate user contact mapping validation.
   */
  public function testDuplicateUserContactValidation(): void {
    $test_user = $this->createUser([], 'duplicate_user');

    // Create an existing mapping.
    $existing_mapping = UserContact::create([
      'user' => $test_user->id(),
      'crm_contact' => $this->testContact->id(),
    ]);
    $existing_mapping->save();

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/add');

    // Try to create a duplicate mapping.
    $edit = [
      'user[0][target_id]' => $test_user->getAccountName() . ' (' . $test_user->id() . ')',
      'crm_contact[0][target_id]' => $this->testContact->label() . ' (' . $this->testContact->id() . ')',
    ];
    $this->submitForm($edit, 'Save');

    // Should show validation error due to unique constraint.
    $uid = $test_user->id();
    $this->assertSession()->pageTextContains("A crm user contact mapping with user referencing entity with ID $uid already exists.");
  }

  /**
   * Tests the delete functionality.
   */
  public function testDeleteUserContactMapping(): void {
    // Create a user contact mapping to delete.
    $test_user = $this->createUser([], 'delete_user');
    $user_contact = UserContact::create([
      'user' => $test_user->id(),
      'crm_contact' => $this->testContact->id(),
    ]);
    $user_contact->save();

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/' . $user_contact->id() . '/delete');
    $this->assertSession()->statusCodeEquals(200);

    // Verify delete confirmation page.
    $this->assertSession()->pageTextContains('Are you sure you want to delete the crm user contact mapping');
    $this->assertSession()->buttonExists('Delete');

    // Confirm deletion.
    $this->submitForm([], 'Delete');

    // Verify entity was deleted.
    $deleted_entity = UserContact::load($user_contact->id());
    $this->assertNull($deleted_entity);

    // Verify redirect to collection page.
    $this->assertSession()->addressEquals('admin/config/crm/user/list');
  }

  /**
   * Tests form behavior with invalid user reference.
   */
  public function testFormWithInvalidUserReference(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/add');

    // Submit form with invalid user reference.
    $edit = [
      'user[0][target_id]' => 'Invalid User (999999)',
      'crm_contact[0][target_id]' => $this->testContact->label() . ' (' . $this->testContact->id() . ')',
    ];
    $this->submitForm($edit, 'Save');

    // Should show validation error.
    $this->assertSession()->pageTextContains('The referenced entity (user: 999999) does not exist.');
  }

  /**
   * Tests form behavior with invalid contact reference.
   */
  public function testFormWithInvalidContactReference(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/add');

    $test_user = $this->createUser([], 'invalid_contact_user');

    // Submit form with invalid contact reference.
    $edit = [
      'user[0][target_id]' => $test_user->getAccountName() . ' (' . $test_user->id() . ')',
      'crm_contact[0][target_id]' => 'Invalid Contact (999999)',
    ];
    $this->submitForm($edit, 'Save');

    // Should show validation error.
    $this->assertSession()->pageTextContains('The referenced entity (crm_contact: 999999) does not exist.');
  }

  /**
   * Tests autocomplete functionality for user field.
   */
  public function testUserFieldAutocomplete(): void {
    $this->drupalLogin($this->adminUser);

    // Create users with different names for autocomplete testing.
    $user1 = $this->createUser([], 'autocomplete_user_1');
    $user2 = $this->createUser([], 'autocomplete_user_2');
    $user3 = $this->createUser([], 'different_name');

    $this->drupalGet('admin/config/crm/user/add');

    // Test that the user field has autocomplete functionality.
    $this->assertSession()->elementExists('css', 'input[data-autocomplete-path]');

    // Test partial user search.
    $autocomplete_path = $this->getSession()->getPage()->find('css', 'input[name="user[0][target_id]"]')->getAttribute('data-autocomplete-path');
    $this->drupalGet($autocomplete_path, ['query' => ['q' => 'autocomplete_user']]);

    $response = $this->getSession()->getPage()->getContent();
    $this->assertStringContainsString('autocomplete_user_1', $response);
    $this->assertStringContainsString('autocomplete_user_2', $response);
    $this->assertStringNotContainsString('different_name', $response);
  }

  /**
   * Tests autocomplete functionality for contact field.
   */
  public function testContactFieldAutocomplete(): void {
    $this->drupalLogin($this->adminUser);

    // Create contacts with different names for autocomplete testing.
    $contact1 = Contact::create([
      'bundle' => 'person',
      'full_name' => [
        'given' => 'Autocomplete',
        'family' => 'Contact1',
      ],
    ]);
    $contact1->save();

    $contact2 = Contact::create([
      'bundle' => 'person',
      'full_name' => [
        'given' => 'Autocomplete',
        'family' => 'Contact2',
      ],
    ]);
    $contact2->save();

    $this->drupalGet('admin/config/crm/user/add');

    // Test that the contact field has autocomplete functionality.
    $this->assertSession()->elementExists('css', 'input[data-autocomplete-path]');

    // Test partial contact search.
    $autocomplete_path = $this->getSession()->getPage()->find('css', 'input[name="crm_contact[0][target_id]"]')->getAttribute('data-autocomplete-path');
    $this->drupalGet($autocomplete_path, ['query' => ['q' => 'Autocomplete Contact1']]);

    $response = $this->getSession()->getPage()->getContent();
    $this->assertStringContainsString('Autocomplete Contact1', $response);
  }

  /**
   * Tests form redirects after submission.
   */
  public function testFormRedirects(): void {
    $this->drupalLogin($this->adminUser);

    $test_user = $this->createUser([], 'redirect_user');

    // Test redirect after creating new user contact mapping.
    $this->drupalGet('admin/config/crm/user/add');
    $edit = [
      'user[0][target_id]' => $test_user->getAccountName() . ' (' . $test_user->id() . ')',
      'crm_contact[0][target_id]' => $this->testContact->label() . ' (' . $this->testContact->id() . ')',
    ];
    $this->submitForm($edit, 'Save');

    // Should redirect to collection page.
    $this->assertSession()->addressEquals('admin/config/crm/user/list');

    // Get the created mapping for editing test.
    $query = \Drupal::entityQuery('crm_user_contact')
      ->condition('user', $test_user->id())
      ->condition('crm_contact', $this->testContact->id())
      ->accessCheck(FALSE);
    $results = $query->execute();
    $mapping_id = reset($results);

    // Test redirect after editing user contact mapping.
    $this->drupalGet('admin/config/crm/user/' . $mapping_id . '/edit');
    $edit = [
      'crm_contact[0][target_id]' => $this->anotherContact->label() . ' (' . $this->anotherContact->id() . ')',
    ];
    $this->submitForm($edit, 'Save');

    // Should redirect to collection page.
    $this->assertSession()->addressEquals('admin/config/crm/user/list');
  }

}
