<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Unit\Form;

use Drupal\Tests\UnitTestCase;
use Drupal\crm\Form\DetailTypeForm;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\crm\Entity\DetailType;
use Drupal\crm\Entity\ContactDetailType;

/**
 * Detail type form test.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Form\DetailTypeForm
 * @uses \Drupal\crm\Form\DetailTypeForm
 */
class DetailTypeFormTest extends UnitTestCase {

  const SAVED_NEW = 1;
  const SAVED_UPDATED = 2;

  /**
   * The form under test.
   *
   * @var \Drupal\crm\Form\DetailTypeForm
   */
  protected $form;

  /**
   * The messenger service.
   *
   * @var \Drupal\Core\Messenger\MessengerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $messenger;

  /**
   * The detail type entity.
   *
   * @var \Drupal\crm\Entity\DetailType|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entity;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $entityTypeManager;

  /**
   * The contact detail type storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $contactDetailTypeStorage;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $container = new ContainerBuilder();

    $string_translation = $this->getStringTranslationStub();
    $container->set('string_translation', $string_translation);

    $this->messenger = $this->createMock(MessengerInterface::class);
    $container->set('messenger', $this->messenger);

    $this->entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $this->contactDetailTypeStorage = $this->createMock(EntityStorageInterface::class);

    $this->entityTypeManager->expects($this->any())
      ->method('getStorage')
      ->with('crm_contact_detail_type')
      ->willReturn($this->contactDetailTypeStorage);

    $container->set('entity_type.manager', $this->entityTypeManager);

    \Drupal::setContainer($container);

    $this->form = DetailTypeForm::create($container);

    $this->entity = $this->createMock(DetailType::class);
    $this->form->setEntity($this->entity);
    $this->form->setEntityTypeManager($this->entityTypeManager);
  }

  /**
   * Tests the create method.
   *
   * @covers ::create
   */
  public function testCreate(): void {
    $container = \Drupal::getContainer();
    $form = DetailTypeForm::create($container);

    $this->assertInstanceOf(DetailTypeForm::class, $form);
  }

  /**
   * Test the form id.
   *
   * @covers ::getFormId
   */
  public function testGetFormId(): void {
    $entity_type = $this->createMock(EntityTypeInterface::class);
    $entity_type->expects($this->once())
      ->method('hasKey')
      ->with('bundle')
      ->willReturn(FALSE);

    $this->entity->expects($this->once())
      ->method('getEntityTypeId')
      ->willReturn('crm_detail_type');
    $this->entity->expects($this->once())
      ->method('getEntityType')
      ->willReturn($entity_type);

    $this->form->setOperation('edit');

    $this->assertEquals('crm_detail_type_edit_form', $this->form->getFormId());
  }

  /**
   * Test the form build method.
   *
   * @covers ::form
   */
  public function testForm(): void {
    // Mock contact detail types for bundles selection.
    $contact_detail_type1 = $this->createMock(ContactDetailType::class);
    $contact_detail_type1->expects($this->any())
      ->method('label')
      ->willReturn('Address');

    $contact_detail_type2 = $this->createMock(ContactDetailType::class);
    $contact_detail_type2->expects($this->any())
      ->method('label')
      ->willReturn('Email');

    $contact_detail_types = [
      'address' => $contact_detail_type1,
      'email' => $contact_detail_type2,
    ];

    $this->contactDetailTypeStorage->expects($this->once())
      ->method('loadMultiple')
      ->willReturn($contact_detail_types);

    // Set up entity expectations.
    $this->entity->expects($this->any())
      ->method('label')
      ->willReturn('Test Detail Type');
    $this->entity->expects($this->any())
      ->method('id')
      ->willReturn('test_detail_type');
    $this->entity->expects($this->any())
      ->method('status')
      ->willReturn(TRUE);
    $this->entity->expects($this->any())
      ->method('isNew')
      ->willReturn(FALSE);
    $this->entity->expects($this->any())
      ->method('get')
      ->willReturnMap([
        ['description', 'Test description'],
        ['bundles', ['address']],
        ['negate', FALSE],
      ]);

    $form_state = $this->createMock(FormStateInterface::class);
    $form = $this->form->form([], $form_state);

    $this->assertIsArray($form);

    // Test label field.
    $this->assertArrayHasKey('label', $form);
    $this->assertEquals('textfield', $form['label']['#type']);
    $this->assertEquals('Test Detail Type', $form['label']['#default_value']);
    $this->assertEquals(255, $form['label']['#maxlength']);
    $this->assertTrue($form['label']['#required']);
    $this->assertArrayHasKey('#description', $form['label']);

    // Test id field (machine name).
    $this->assertArrayHasKey('id', $form);
    $this->assertEquals('machine_name', $form['id']['#type']);
    $this->assertEquals('test_detail_type', $form['id']['#default_value']);
    $this->assertArrayHasKey('#machine_name', $form['id']);
    $this->assertEquals('\Drupal\crm\Entity\DetailType::load', $form['id']['#machine_name']['exists']);
    // Should be disabled for existing entity.
    $this->assertTrue($form['id']['#disabled']);

    // Test status field.
    $this->assertArrayHasKey('status', $form);
    $this->assertEquals('checkbox', $form['status']['#type']);
    $this->assertTrue($form['status']['#default_value']);

    // Test description field.
    $this->assertArrayHasKey('description', $form);
    $this->assertEquals('textarea', $form['description']['#type']);
    $this->assertEquals('Test description', $form['description']['#default_value']);
    $this->assertArrayHasKey('#description', $form['description']);

    // Test bundles field.
    $this->assertArrayHasKey('bundles', $form);
    $this->assertEquals('checkboxes', $form['bundles']['#type']);
    $this->assertEquals(['address' => 'Address', 'email' => 'Email'], $form['bundles']['#options']);
    $this->assertEquals(['address'], $form['bundles']['#default_value']);
    $this->assertTrue($form['bundles']['#multiple']);
    $this->assertArrayHasKey('#description', $form['bundles']);

    // Test negate field.
    $this->assertArrayHasKey('negate', $form);
    $this->assertEquals('checkbox', $form['negate']['#type']);
    $this->assertFalse($form['negate']['#default_value']);
    $this->assertArrayHasKey('#description', $form['negate']);
  }

  /**
   * Test the form build method with new entity.
   *
   * @covers ::form
   */
  public function testFormWithNewEntity(): void {
    // Mock contact detail types.
    $contact_detail_types = [];
    $this->contactDetailTypeStorage->expects($this->once())
      ->method('loadMultiple')
      ->willReturn($contact_detail_types);

    // Set up new entity expectations.
    $this->entity->expects($this->any())
      ->method('label')
      ->willReturn('');
    $this->entity->expects($this->any())
      ->method('id')
      ->willReturn('');
    $this->entity->expects($this->any())
      ->method('status')
      ->willReturn(TRUE);
    $this->entity->expects($this->any())
      ->method('isNew')
      ->willReturn(TRUE);
    $this->entity->expects($this->any())
      ->method('get')
      ->willReturnMap([
        ['description', ''],
        ['bundles', []],
        ['negate', FALSE],
      ]);

    $form_state = $this->createMock(FormStateInterface::class);
    $form = $this->form->form([], $form_state);

    // Test that id field is not disabled for new entity.
    $this->assertFalse($form['id']['#disabled']);
  }

  /**
   * Test the save method with new entity.
   *
   * @covers ::save
   */
  public function testSaveNewEntity(): void {
    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->expects($this->once())
      ->method('getValue')
      ->with('bundles')
      ->willReturn(['address' => 'address', 'email' => 0, 'telephone' => 'telephone']);

    $this->entity->expects($this->once())
      ->method('set')
      ->with('bundles', [0 => 'address', 2 => 'telephone']);

    $this->entity->expects($this->any())
      ->method('label')
      ->willReturn('Test Detail Type');
    $this->entity->expects($this->once())
      ->method('save')
      ->willReturn(self::SAVED_NEW);

    $url = $this->createMock('Drupal\Core\Url');
    $this->entity->expects($this->once())
      ->method('toUrl')
      ->with('collection')
      ->willReturn($url);

    $form_state->expects($this->once())
      ->method('setRedirectUrl')
      ->with($url);

    $this->messenger->expects($this->once())
      ->method('addStatus')
      ->with($this->getStringTranslationStub()->translate('Created new detail type %label.', ['%label' => 'Test Detail Type']));

    $result = $this->form->save([], $form_state);
    $this->assertEquals(self::SAVED_NEW, $result);
  }

  /**
   * Test the save method with existing entity.
   *
   * @covers ::save
   */
  public function testSaveExistingEntity(): void {
    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->expects($this->once())
      ->method('getValue')
      ->with('bundles')
      ->willReturn(['address' => 'address', 'email' => 'email']);

    $this->entity->expects($this->once())
      ->method('set')
      ->with('bundles', ['address', 'email']);

    $this->entity->expects($this->any())
      ->method('label')
      ->willReturn('Updated Detail Type');
    $this->entity->expects($this->once())
      ->method('save')
      ->willReturn(self::SAVED_UPDATED);

    $url = $this->createMock('Drupal\Core\Url');
    $this->entity->expects($this->once())
      ->method('toUrl')
      ->with('collection')
      ->willReturn($url);

    $form_state->expects($this->once())
      ->method('setRedirectUrl')
      ->with($url);

    $this->messenger->expects($this->once())
      ->method('addStatus')
      ->with($this->getStringTranslationStub()->translate('Updated detail type %label.', ['%label' => 'Updated Detail Type']));

    $result = $this->form->save([], $form_state);
    $this->assertEquals(self::SAVED_UPDATED, $result);
  }

  /**
   * Test the save method filters bundles correctly.
   *
   * @covers ::save
   */
  public function testSaveFiltersBundles(): void {
    $form_state = $this->createMock(FormStateInterface::class);
    $form_state->expects($this->once())
      ->method('getValue')
      ->with('bundles')
      ->willReturn(['address' => 'address', 'email' => 0, 'telephone' => '', 'fax' => 'fax']);

    // Should filter out falsy values (0 and empty string)
    // and keep only truthy values.
    // Note: array_filter preserves keys, so we expect
    // [0 => 'address', 3 => 'fax'].
    $this->entity->expects($this->once())
      ->method('set')
      ->with('bundles', [0 => 'address', 3 => 'fax']);

    $this->entity->expects($this->any())
      ->method('label')
      ->willReturn('Test Detail Type');
    $this->entity->expects($this->once())
      ->method('save')
      ->willReturn(self::SAVED_NEW);

    $url = $this->createMock('Drupal\Core\Url');
    $this->entity->expects($this->once())
      ->method('toUrl')
      ->with('collection')
      ->willReturn($url);

    $form_state->expects($this->once())
      ->method('setRedirectUrl')
      ->with($url);

    $this->messenger->expects($this->once())
      ->method('addStatus');

    $this->form->save([], $form_state);
  }

}
