<?php

declare(strict_types=1);

namespace Drupal\crm;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityListBuilder;
use Drupal\Core\Url;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a list controller for the CRM contact detail entity type.
 */
class ContactDetailListBuilder extends EntityListBuilder {

  /**
   * The entity storage class for the contact entity.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $contactStorage;

  /**
   * Constructs a new ContactDetailListBuilder object.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The entity storage class.
   * @param \Drupal\Core\Entity\EntityStorageInterface $contact_storage
   *   The entity storage class for the contact entity.
   */
  public function __construct(EntityTypeInterface $entity_type, EntityStorageInterface $storage, EntityStorageInterface $contact_storage) {
    parent::__construct($entity_type, $storage);
    $this->contactStorage = $contact_storage;
  }

  /**
   * {@inheritdoc}
   */
  final public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new self(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('entity_type.manager')->getStorage('crm_contact'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader(): array {
    $header['id'] = $this->t('ID');
    $header['label'] = $this->t('Label');
    $header['type'] = $this->t('Type');
    $header['reference'] = $this->t('Reference');
    $header['status'] = $this->t('Status');
    $header['created'] = $this->t('Created');
    $header['changed'] = $this->t('Updated');
    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity): array {
    $bundle = $entity->get('bundle')->entity;
    /** @var \Drupal\crm\CRMContactDetailInterface $entity */
    $row['id'] = $entity->id();
    $row['label'] = $entity->label();
    $row['type'] = $bundle->label();
    $row['references'] = $this->getReferenceCount($entity->id(), $bundle->id());
    $row['status'] = $entity->get('status')->value ? $this->t('Enabled') : $this->t('Disabled');
    $row['created']['data'] = $entity->get('created')->view(['label' => 'hidden']);
    $row['changed']['data'] = $entity->get('changed')->view(['label' => 'hidden']);
    return $row + parent::buildRow($entity);
  }

  /**
   * {@inheritdoc}
   */
  public function render() {
    $build = parent::render();

    $build['table']['#empty'] = $this->t(
      'No crm detail types available. <a href=":link">Add crm detail type</a>.',
      [':link' => Url::fromRoute('entity.crm_detail_type.add_form')->toString()]
    );
    $total = $this->getStorage()
      ->getQuery()
      ->accessCheck(FALSE)
      ->count()
      ->execute();

    $build['summary']['#markup'] = $this->t('Total details: @total', ['@total' => $total]);

    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function getDefaultOperations(EntityInterface $entity) {
    $operations = parent::getDefaultOperations($entity);
    // Place the edit operation after the operations added by field_ui.module
    // which have the weights 15, 20, 25.
    if (isset($operations['edit'])) {
      $operations['edit']['weight'] = 30;
    }
    return $operations;
  }

  /**
   * Get the number of references to a contact detail.
   *
   * @param mixed $id
   *   The ID of the contact detail.
   * @param string $bundle
   *   The bundle of the contact detail.
   *
   * @return int
   *   The number of references to the contact detail.
   */
  protected function getReferenceCount($id, string $bundle) : int {
    $query = $this->contactStorage->getQuery();

    switch ($bundle) {
      case 'email':
        $query->condition('emails', $id);
        break;

      case 'telephone':
        $query->condition('telephones', $id);
        break;

      case 'address':
        $query->condition('addresses', $id);
        break;
    }

    return $query->accessCheck(FALSE)->count()->execute();
  }

}
