<?php

namespace Drupal\crm\ContextProvider;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Plugin\Context\Context;
use Drupal\Core\Plugin\Context\ContextProviderInterface;
use Drupal\Core\Plugin\Context\EntityContext;
use Drupal\Core\Plugin\Context\EntityContextDefinition;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\crm\Entity\Contact;
use Drupal\Core\StringTranslation\StringTranslationTrait;

/**
 * Sets the current contact as a context on crm_contact routes.
 */
class ContactRouteContext implements ContextProviderInterface {

  use StringTranslationTrait;

  /**
   * The route match object.
   *
   * @var \Drupal\Core\Routing\RouteMatchInterface
   */
  protected $routeMatch;

  /**
   * Constructs a new ContactRouteContext.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface $route_match
   *   The route match object.
   */
  public function __construct(RouteMatchInterface $route_match) {
    $this->routeMatch = $route_match;
  }

  /**
   * {@inheritdoc}
   */
  public function getRuntimeContexts(array $unqualified_context_ids) {
    $result = [];
    $context_definition = EntityContextDefinition::create('crm_contact')->setRequired(FALSE);
    $value = NULL;
    if (($route_object = $this->routeMatch->getRouteObject())) {
      $route_contexts = $route_object->getOption('parameters');
      // Check for a crm_contact revision parameter first.
      if (isset($route_contexts['crm_contact_revision']) && $revision = $this->routeMatch->getParameter('crm_contact_revision')) {
        $value = $revision;
      }
      elseif (isset($route_contexts['crm_contact']) && $contact = $this->routeMatch->getParameter('crm_contact')) {
        $value = $contact;
      }
      elseif ($this->routeMatch->getRouteName() == 'crm_contact.add') {
        $contact_type = $this->routeMatch->getParameter('crm_contact_type');
        $value = Contact::create(['bundle' => $contact_type->id()]);
      }
    }

    $cacheability = new CacheableMetadata();
    $cacheability->setCacheContexts(['route']);

    $context = new Context($context_definition, $value);
    $context->addCacheableDependency($cacheability);
    $result['crm_contact'] = $context;

    return $result;
  }

  /**
   * {@inheritdoc}
   */
  public function getAvailableContexts() {
    $context = EntityContext::fromEntityTypeId('crm_contact', $this->t('Contact from URL'));
    return ['crm_contact' => $context];
  }

}
