<?php

declare(strict_types=1);

namespace Drupal\crm\Entity;

use Drupal\Core\Entity\Attribute\ContentEntityType;
use Drupal\Core\Entity\ContentEntityDeleteForm;
use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Entity\EntityChangedTrait;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Entity\RevisionableContentEntityBase;
use Drupal\Core\Entity\RevisionLogEntityTrait;
use Drupal\Core\Entity\Routing\AdminHtmlRouteProvider;
use Drupal\Core\Field\BaseFieldDefinition;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\crm\ContactDetailAccessControlHandler;
use Drupal\crm\ContactDetailListBuilder;
use Drupal\crm\CrmContactDetailInterface;
use Drupal\views\EntityViewsData;

/**
 * CRM contact detail.
 *
 * A contact can have many details: email, address, phone number, etc.
 */
#[ContentEntityType(
  id: 'crm_contact_detail',
  label: new TranslatableMarkup('CRM Contact Detail'),
  label_collection: new TranslatableMarkup('CRM Contact Details'),
  label_singular: new TranslatableMarkup('crm contact detail'),
  label_plural: new TranslatableMarkup('crm contact details'),
  label_count: [
    'singular' => '@count crm contact detail',
    'plural' => '@count crm contact details',
  ],
  bundle_label: new TranslatableMarkup('CRM Contact Detail type'),
  handlers: [
    'access' => ContactDetailAccessControlHandler::class,
    'list_builder' => ContactDetailListBuilder::class,
    'views_data' => EntityViewsData::class,
    'form' => [
      'default' => ContentEntityForm::class,
      'delete' => ContentEntityDeleteForm::class,
    ],
    'route_provider' => [
      'html' => AdminHtmlRouteProvider::class,
    ],
  ],
    base_table: 'crm_contact_detail',
    revision_table: 'crm_contact_detail_revision',
    show_revision_ui: TRUE,
    admin_permission: 'administer crm',
    entity_keys: [
      'id' => 'id',
      'revision' => 'revision_id',
      'bundle' => 'bundle',
      'label' => 'id',
      'uuid' => 'uuid',
      'status' => 'status',
    ],
    revision_metadata_keys: [
      'revision_user' => 'revision_uid',
      'revision_created' => 'revision_timestamp',
      'revision_log_message' => 'revision_log',
    ],
    bundle_entity_type: 'crm_contact_detail_type',
    field_ui_base_route: 'entity.crm_contact_detail_type.edit_form',
    translatable: FALSE,
    links: [
      'collection' => '/admin/content/crm/detail',
    ],
)]
class ContactDetail extends RevisionableContentEntityBase implements CrmContactDetailInterface {

  use EntityChangedTrait;
  use RevisionLogEntityTrait;

  /**
   * {@inheritdoc}
   */
  public static function baseFieldDefinitions(EntityTypeInterface $entity_type): array {

    $fields = parent::baseFieldDefinitions($entity_type);

    $fields['type'] = BaseFieldDefinition::create('entity_reference')
      ->setRevisionable(TRUE)
      ->setLabel(t('Type'))
      ->setSetting('target_type', 'crm_detail_type')
      ->setDisplayOptions('form', [
        'type' => 'options_select',
        'weight' => -10,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'entity_reference_label',
        'weight' => 10,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['status'] = BaseFieldDefinition::create('boolean')
      ->setRevisionable(TRUE)
      ->setLabel(t('Status'))
      ->setDefaultValue(TRUE)
      ->setSetting('on_label', 'Enabled')
      ->setDisplayOptions('form', [
        'type' => 'boolean_checkbox',
        'settings' => [
          'display_label' => FALSE,
        ],
        'weight' => 0,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('view', [
        'type' => 'boolean',
        'label' => 'above',
        'weight' => 0,
        'settings' => [
          'format' => 'enabled-disabled',
        ],
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['crm_contact'] = BaseFieldDefinition::create('entity_reference')
      ->setLabel(t('CRM Contact'))
      ->setDescription(t('The CRM Contact that this detail belongs to.'))
      ->setSetting('target_type', 'crm_contact');

    $fields['created'] = BaseFieldDefinition::create('created')
      ->setLabel(t('Created on'))
      ->setDescription(t('The time that the crm contact detail was created.'))
      ->setDisplayOptions('view', [
        'label' => 'above',
        'type' => 'timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('form', TRUE)
      ->setDisplayOptions('form', [
        'type' => 'datetime_timestamp',
        'weight' => 20,
      ])
      ->setDisplayConfigurable('view', TRUE);

    $fields['changed'] = BaseFieldDefinition::create('changed')
      ->setLabel(t('Changed'))
      ->setDescription(t('The time that the crm contact detail was last edited.'));

    return $fields;
  }

}
