<?php

namespace Drupal\crm\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBundleBase;
use Drupal\Core\Entity\Attribute\ConfigEntityType;
use Drupal\Core\Entity\EntityDeleteForm;
use Drupal\Core\Entity\Routing\AdminHtmlRouteProvider;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\crm\RelationshipTypeListBuilder;
use Drupal\crm\Form\RelationshipTypeForm;

use Drupal\crm\CrmRelationshipTypeInterface;

/**
 * Defines the CRM Relationship type configuration entity.
 */
#[ConfigEntityType(
  id: 'crm_relationship_type',
  label: new TranslatableMarkup('CRM Relationship type'),
  label_collection: new TranslatableMarkup('CRM Relationship types'),
  label_singular: new TranslatableMarkup('crm relationship type'),
  label_plural: new TranslatableMarkup('crm relationships types'),
  label_count: [
    'singular' => '@count crm relationships type',
    'plural' => '@count crm relationships types',
  ],
  handlers: [
    'form' => [
      'add' => RelationshipTypeForm::class,
      'edit' => RelationshipTypeForm::class,
      'delete' => EntityDeleteForm::class,
    ],
    'list_builder' => RelationshipTypeListBuilder::class,
    'route_provider' => [
      'html' => AdminHtmlRouteProvider::class,
    ],
  ],
  admin_permission: 'administer crm',
  bundle_of: 'crm_relationship',
  config_prefix: 'crm_relationship_type',
  entity_keys: [
    'id' => 'id',
    'label' => 'label',
    'uuid' => 'uuid',
  ],
  links: [
    'add-form' => '/admin/structure/crm/relationship_types/add',
    'edit-form' => '/admin/structure/crm/relationship_types/manage/{crm_relationship_type}',
    'delete-form' => '/admin/structure/crm/relationship_types/manage/{crm_relationship_type}/delete',
    'collection' => '/admin/structure/crm/relationship_types',
  ],
  config_export: [
    'id',
    'label',
    'description',
    'asymmetric',
    'uuid',
    'contact_type_a',
    'contact_type_b',
    'label_a',
    'label_b',
  ],
)]
class RelationshipType extends ConfigEntityBundleBase implements CrmRelationshipTypeInterface {

  /**
   * The machine name of this crm relationship type.
   *
   * @var string
   */
  protected $id;

  /**
   * The human-readable name of the crm relationship type.
   *
   * @var string
   */
  protected $label;

  /**
   * The description of the crm relationship type.
   *
   * @var string
   */
  protected $description;

  /**
   * Is the bundle and label the same for both sides of the relationship?
   *
   * @var bool
   */
  protected $asymmetric;

  /**
   * The human-readable name of the crm relationship type.
   *
   * @var string
   */
  protected $label_a;

  /**
   * The human-readable name of the crm relationship type.
   *
   * @var string
   */
  protected $label_b;

  /**
   * The human-readable name of the crm relationship type.
   *
   * @var string
   */
  protected $contact_type_a;

  /**
   * The human-readable name of the crm relationship type.
   *
   * @var string
   */
  protected $contact_type_b;

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->description ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function setDescription($description) {
    $this->description = $description;

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies() {
    parent::calculateDependencies();

    if ($type_a = $this->get('contact_type_a')) {
      $this->addDependency('config', 'crm.crm_contact_type.' . $type_a);
    }
    if ($type_b = $this->get('contact_type_b')) {
      $this->addDependency('config', 'crm.crm_contact_type.' . $type_b);
    }

    return $this;
  }

}
