<?php

namespace Drupal\crm\Form;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\Core\Datetime\DateFormatterInterface;
use Drupal\Core\Entity\ContentEntityForm;
use Drupal\Core\Entity\EntityRepositoryInterface;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form controller for the contact entity edit forms.
 */
class ContactForm extends ContentEntityForm {

  /**
   * The date formatter service.
   *
   * @var \Drupal\Core\Datetime\DateFormatterInterface
   */
  protected $dateFormatter;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected $currentUser;

  /**
   * Constructs a ContactForm object.
   *
   * @param \Drupal\Core\Entity\EntityRepositoryInterface $entity_repository
   *   The entity repository.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   * @param \Drupal\Core\Datetime\DateFormatterInterface $date_formatter
   *   The date formatter service.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface|null $entity_type_bundle_info
   *   The entity type bundle service.
   */
  public function __construct(
    EntityRepositoryInterface $entity_repository,
    TimeInterface $time,
    DateFormatterInterface $date_formatter,
    AccountInterface $current_user,
    ?EntityTypeBundleInfoInterface $entity_type_bundle_info = NULL,
  ) {

    parent::__construct($entity_repository, $entity_type_bundle_info, $time);
    $this->dateFormatter = $date_formatter;
    $this->currentUser = $current_user;
  }

  /**
   * {@inheritdoc}
   */
  final public static function create(ContainerInterface $container) {
    return new self(
      $container->get('entity.repository'),
      $container->get('datetime.time'),
      $container->get('date.formatter'),
      $container->get('current_user'),
      $container->get('entity_type.bundle.info'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    /** @var \Drupal\crm\CrmContactInterface $contact */
    $contact = $this->entity;
    $contact_type = $contact->get('bundle')->entity;
    $date = $contact_type->get('date');
    $form = parent::form($form, $form_state);

    $form['advanced']['#attributes']['class'][] = 'entity-meta';

    $form['meta'] = [
      '#type' => 'details',
      '#group' => 'advanced',
      '#weight' => -10,
      '#title' => $this->t('Status'),
      '#attributes' => ['class' => ['entity-meta__header']],
      '#tree' => TRUE,
      '#access' => $this->currentUser->hasPermission('administer crm'),
    ];
    $form['meta']['published'] = [
      '#type' => 'item',
      '#markup' => $contact->get('status')->value ? $this->t('Active') : $this->t('Inactive'),
      '#access' => !$contact->isNew(),
      '#wrapper_attributes' => ['class' => ['entity-meta__title']],
    ];
    $form['meta']['changed'] = [
      '#type' => 'item',
      '#title' => $this->t('Last saved'),
      '#markup' => !$contact->isNew() ? $this->dateFormatter->format($contact->getChangedTime(), 'short') : $this->t('Not saved yet'),
      '#wrapper_attributes' => ['class' => ['entity-meta__last-saved']],
    ];

    $form['meta']['status'] = &$form['status'];
    unset($form['status']);

    if (isset($form['created'])) {
      $form['created']['#weight'] = 200;
      $form['meta']['created'] = &$form['created'];
      unset($form['created']);
    }

    $form['age'] = [
      '#type' => 'details',
      '#title' => $this->t('Age'),
      '#group' => 'advanced',
    ];

    if ($date && !empty($date['start_date']['label']) && isset($form['start_date']['widget'])) {
      $form['start_date']['widget'][0]['#title'] = $date['start_date']['label'];
      $form['start_date']['widget']['#title'] = $date['start_date']['label'];
      $form['start_date']['widget'][0]['value']['#title'] = $date['start_date']['label'];
    }

    if ($date && !empty($date['start_date']['description']) && isset($form['start_date']['widget'])) {
      $form['start_date']['widget'][0]['#description'] = $date['start_date']['description'];
      $form['start_date']['widget']['#description'] = $date['start_date']['description'];
      $form['start_date']['widget'][0]['value']['#description'] = $date['start_date']['description'];
    }

    if ($date && !empty($date['end_date']['label']) && isset($form['end_date']['widget'])) {
      $form['end_date']['widget'][0]['#title'] = $date['end_date']['label'];
      $form['end_date']['widget']['#title'] = $date['end_date']['label'];
      $form['end_date']['widget'][0]['value']['#title'] = $date['end_date']['label'];
    }

    if ($date && !empty($date['end_date']['description']) && isset($form['end_date']['widget'])) {
      $form['end_date']['widget'][0]['#description'] = $date['end_date']['description'];
      $form['end_date']['widget']['#description'] = $date['end_date']['description'];
      $form['end_date']['widget'][0]['value']['#description'] = $date['end_date']['description'];
    }

    $form['age']['start_date'] = &$form['start_date'];
    unset($form['start_date']);

    $form['age']['end_date'] = &$form['end_date'];
    unset($form['end_date']);

    // Add a custom submit handler to the entity form.
    $form['revision']['#default_value'] = TRUE;

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {

    $contact = $this->getEntity();
    // Set new Revision.
    $contact->setNewRevision(TRUE);

    $result = parent::save($form, $form_state);

    $message_arguments = ['%label' => $contact->toLink()->toString()];
    $logger_arguments = [
      '%label' => $contact->label(),
      'link' => $contact->toLink($this->t('View'))->toString(),
    ];

    switch ($result) {
      case SAVED_NEW:
        $this->messenger()->addStatus($this->t('New contact %label has been created.', $message_arguments));
        $this->logger('crm_contact')->notice('Created new contact %label', $logger_arguments);
        break;

      case SAVED_UPDATED:
        $this->messenger()->addStatus($this->t('The contact %label has been updated.', $message_arguments));
        $this->logger('crm_contact')->notice('Updated contact %label.', $logger_arguments);
        break;
    }

    $form_state->setRedirect('entity.crm_contact.canonical', [
      'crm_contact' => $contact->id(),
    ]);

    if ($contact->id()) {
      $form_state->setValue('id', $contact->id());
      $form_state->set('id', $contact->id());
      if ($contact->access('view')) {
        $form_state->setRedirect(
          'entity.crm_contact.canonical',
          ['crm_contact' => $contact->id()]
        );
      }
      else {
        $form_state->setRedirect('<front>');
      }

    }

    return $result;
  }

}
