<?php

namespace Drupal\crm\Form;

use Drupal\Core\Entity\BundleEntityFormBase;
use Drupal\Core\Entity\EntityTypeBundleInfoInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Form handler for crm relationship type forms.
 */
class RelationshipTypeForm extends BundleEntityFormBase {

  const SAVED_NEW = 1;
  const SAVED_UPDATED = 2;

  /**
   * The entity type bundle info service.
   *
   * @var \Drupal\Core\Entity\EntityTypeBundleInfoInterface
   */
  protected $entityTypeBundleInfo;

  /**
   * Constructs a RelationshipTypeForm object.
   *
   * @param \Drupal\Core\Entity\EntityTypeBundleInfoInterface $entity_type_bundle_info
   *   The entity type bundle info service.
   */
  public function __construct(EntityTypeBundleInfoInterface $entity_type_bundle_info) {
    $this->entityTypeBundleInfo = $entity_type_bundle_info;
  }

  /**
   * {@inheritdoc}
   */
  final public static function create(ContainerInterface $container) {
    return new self(
      $container->get('entity_type.bundle.info')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function form(array $form, FormStateInterface $form_state) {
    $form = parent::form($form, $form_state);

    $entity_type = $this->entity;
    if ($this->operation == 'edit') {
      $form['#title'] = $this->t('Edit %label crm relationship type', ['%label' => $entity_type->label()]);
    }

    $form['label'] = [
      '#title' => $this->t('Label'),
      '#type' => 'textfield',
      '#default_value' => $entity_type->label(),
      '#description' => $this->t('The human-readable name of this crm relationship type.'),
      '#required' => TRUE,
      '#size' => 30,
    ];

    $form['id'] = [
      '#type' => 'machine_name',
      '#default_value' => $entity_type->id(),
      '#maxlength' => EntityTypeInterface::BUNDLE_MAX_LENGTH,
      '#machine_name' => [
        'exists' => ['Drupal\crm\Entity\RelationshipType', 'load'],
        'source' => ['label'],
      ],
      '#description' => $this->t('A unique machine-readable name for this crm relationship type. It must only contain lowercase letters, numbers, and underscores.'),
    ];

    $form['description'] = [
      '#title' => $this->t('Description'),
      '#type' => 'textarea',
      '#default_value' => $entity_type->get('description'),
      '#description' => $this->t('A description of this crm relationship type.'),
    ];

    $form['asymmetric'] = [
      '#title' => $this->t('Asymmetric'),
      '#type' => 'checkbox',
      '#default_value' => $entity_type->isNew() ? 1 : $entity_type->get('asymmetric'),
      '#description' => $this->t('Check this box if the relationship is asymmetric.'),
    ];

    $form['label_a'] = [
      '#title' => $this->t('Contact A label'),
      '#type' => 'textfield',
      '#default_value' => $entity_type->get('label_a'),
      '#description' => $this->t('The human-readable name of this crm relationship type from contact A to contact B.'),
      '#required' => TRUE,
      '#size' => 30,
    ];

    $form['contact_type_a'] = [
      '#title' => $this->t('Contact A type'),
      '#type' => 'select',
      '#options' => $this->getContactTypeOptions(),
      '#default_value' => $entity_type->get('contact_type_a'),
      '#description' => $this->t('The contact type for the first contact in the relationship.'),
      '#required' => TRUE,
    ];

    $form['label_b'] = [
      '#title' => $this->t('Contact B label'),
      '#type' => 'textfield',
      '#default_value' => $entity_type->get('label_b'),
      '#description' => $this->t('The human-readable name of this crm relationship type from contact A to contact B.'),
      '#required' => TRUE,
      '#size' => 30,
      '#states' => [
        'visible' => [
          ':input[name="asymmetric"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['contact_type_b'] = [
      '#title' => $this->t('Contact B type'),
      '#type' => 'select',
      '#options' => $this->getContactTypeOptions(),
      '#default_value' => $entity_type->get('contact_type_b'),
      '#description' => $this->t('The contact type for the second contact in the relationship.'),
      '#required' => TRUE,
      '#states' => [
        'visible' => [
          ':input[name="asymmetric"]' => ['checked' => TRUE],
        ],
      ],
    ];

    return $this->protectBundleIdElement($form);
  }

  /**
   * Returns a list of contact types.
   */
  protected function getContactTypeOptions() {
    $crm_contact_type = $this->entityTypeBundleInfo->getBundleInfo('crm_contact');
    $options = [];
    foreach ($crm_contact_type as $type => $contact) {
      $options[$type] = $contact['label'];
    }

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  protected function actions(array $form, FormStateInterface $form_state) {
    $actions = parent::actions($form, $form_state);
    $actions['submit']['#value'] = $this->t('Save relationship type');

    return $actions;
  }

  /**
   * {@inheritdoc}
   */
  public function save(array $form, FormStateInterface $form_state) {
    $entity_type = $this->entity;
    $entity_type
      ->set('id', trim($entity_type->id()))
      ->set('label', trim($entity_type->label()));

    if (!$entity_type->get('asymmetric')) {
      $entity_type->set('label_b', $entity_type->get('label_a'));
      $entity_type->set('contact_type_b', $entity_type->get('contact_type_a'));
    }

    $status = $entity_type->save();

    $t_args = ['%name' => $entity_type->label()];
    if ($status == self::SAVED_UPDATED) {
      $message = $this->t('The crm relationship type %name has been updated.', $t_args);
    }
    elseif ($status == self::SAVED_NEW) {
      $message = $this->t('The crm relationship type %name has been added.', $t_args);
    }
    $this->messenger()->addStatus($message);

    $form_state->setRedirectUrl($entity_type->toUrl('collection'));

    return $status;
  }

}
