<?php

namespace Drupal\crm\Hook;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Session\AccountInterface;
use Drupal\crm\Event\CrmUserContactEvent;
use Drupal\user\UserDataInterface;
use Drupal\user\UserInterface;
use Symfony\Contracts\EventDispatcher\EventDispatcherInterface;
use Drupal\Core\Session\AccountProxyInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\name\NameFormatterInterface;

/**
 * Hooks relating to users.
 */
class UserHooks {
  use StringTranslationTrait;

  public function __construct(
    protected ConfigFactoryInterface $configFactory,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected EventDispatcherInterface $eventDispatcher,
    protected UserDataInterface $userData,
    protected AccountProxyInterface $currentUser,
    protected NameFormatterInterface $nameFormatter,
  ) {}

  /**
   * Implements hook_ENTITY_TYPE_insert().
   */
  #[Hook('user_insert')]
  public function userInsert(UserInterface $user) {

    $auto_create_crm_user_contact = $this->configFactory
      ->get('crm.user_contact.settings')
      ->get('auto_create_crm_user_contact');
    if (!$auto_create_crm_user_contact) {
      return;
    }

    $crm_user_contact = $this->entityTypeManager
      ->getStorage('crm_user_contact')
      ->create(['user' => $user->id()]);
    $event = new CrmUserContactEvent($crm_user_contact);
    $this->eventDispatcher->dispatch($event, CrmUserContactEvent::EVENT_NAME);
    $crm_user_contact = $event->getCrmUserContact();
    $contact = $crm_user_contact->getContact();

    if ($contact == NULL) {
      return NULL;
    }
    if ($contact->get('bundle')->target_id != 'person') {
      return NULL;
    }

    $crm_user_contact->save();
  }

  /**
   * Implements hook_user_format_name_alter().
   */
  #[Hook('user_format_name_alter')]
  public function userContactFormatNameAlter(&$name, AccountInterface $account) {
    $override = $this->configFactory->get('crm.user_contact.settings')->get('display_name');
    if (!$override) {
      return;
    }
    $crm_users_contact = $this->entityTypeManager
      ->getStorage('crm_user_contact')
      ->loadByProperties(['user' => $account->id()]);
    if ($crm_users_contact == NULL) {
      return;
    }
    /** @var \Drupal\crm\Entity\UserContact */
    $crm_user_contact = reset($crm_users_contact);
    $contact = $crm_user_contact->getContact();
    $name = $contact->label();

    $user_name_format = $this->userData->get('crm', $account->id(), 'name_format');
    if (!$user_name_format) {
      return;
    }
    if (!$account->hasPermission('crm user alter display name')) {
      return;
    }

    $name_array = $contact->get('full_name')->getValue();
    if (empty($name_array)) {
      return;
    }
    $name_array = $name_array[0];
    $name_array['preferred'] = $contact->get('preferred_name')->value;
    $name_array['alternative'] = array_map(function ($alias) {
      return $alias['value'];
    }, $contact->get('aliases')->getValue());
    $name_array['alternative'] = implode(', ', $name_array['alternative']);

    $formatted_name = $this->nameFormatter->format($name_array, $user_name_format);

    $name = html_entity_decode($formatted_name, ENT_QUOTES | ENT_HTML5, 'UTF-8');

  }

  /**
   * Implements hook_form_FORM_ID_alter().
   */
  #[Hook('form_user_form_alter')]
  public function userFormAlter(&$form, FormStateInterface $form_state) {
    $user_contact_settings = $this->configFactory->get('crm.user_contact.settings');

    if (!$user_contact_settings->get('display_name')) {
      return;
    }

    $form_user = $form_state->getFormObject()->getEntity();

    $user_contact = $this->entityTypeManager
      ->getStorage('crm_user_contact')
      ->loadByProperties(['user' => $form_user->id()]);
    if ($user_contact == NULL) {
      return;
    }

    $user_contact = reset($user_contact);

    $same_user = $this->currentUser->id() == $form_user->id();
    if (!$same_user) {
      return;
    }

    if (!$this->currentUser->hasPermission('crm user alter display name')) {
      return;
    }

    $format_options = $this->entityTypeManager->getStorage('name_format')->loadMultiple();
    $format_options = array_map(function ($format) {
      return $format->label();
    }, $format_options);

    $person_contact_type = $user_contact->getContact()->get('bundle')->entity;
    $system_format = $person_contact_type->getThirdPartySetting('crm', 'name_format', 'default');
    $format_options = ['_none' => $this->t('System (@system)', ['@system' => $system_format])] + $format_options;

    $user_name_format = $this->userData->get('crm', $form_user->id(), 'name_format');

    $form['crm'] = [
      '#type' => 'details',
      '#title' => $this->t('CRM'),
      '#open' => TRUE,
    ];

    $form['crm']['name_format'] = [
      '#type' => 'select',
      '#title' => $this->t('Name Format'),
      '#options' => $format_options,
      '#default_value' => $user_name_format,
    ];

    $form['actions']['submit']['#submit'][] = [static::class, 'userFormAlterSubmit'];
  }

  /**
   * Static submit handler for user form alter.
   */
  public static function userFormAlterSubmit(array &$form, FormStateInterface $form_state) {

    $form_user = $form_state->getFormObject()->getEntity();
    $user_name_format = $form_state->getValue('name_format');

    /** @var \Drupal\user\UserDataInterface $user_data */
    $user_data = \Drupal::service('user.data');

    if ($user_name_format == '_none') {
      $user_data->delete('crm', $form_user->id(), 'name_format');
      return;
    }
    $user_data->set('crm', $form_user->id(), 'name_format', $user_name_format);
  }

  /**
   * Implements hook_ENTITY_TYPE_delete().
   */
  #[Hook('user_delete')]
  public function userDelete(UserInterface $user) {
    $crm_users_contact = $this->entityTypeManager
      ->getStorage('crm_user_contact')
      ->loadByProperties(['user' => $user->id()]);
    if ($crm_users_contact != NULL) {
      $crm_user_contact = reset($crm_users_contact);
      $crm_user_contact->delete();
    }
  }

}
