<?php

namespace Drupal\Tests\crm\Functional;

use Drupal\crm\Entity\Contact;

/**
 * Tests the creation of contacts.
 *
 * @group crm
 */
class ContactCreationTest extends ContactTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'dblog',
    'crm_test_exception',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $web_user = $this->drupalCreateUser([
      'view any crm_contact',
      'create crm_contact',
    ]);
    $this->drupalLogin($web_user);
  }

  /**
   * Tests that a contact can be created.
   */
  public function testContactCreation() {
    $this->drupalGet('crm/contact/add');
    $this->assertSession()->statusCodeEquals(200);

    // Create a contact.
    $edit = [];
    $edit['name[0][value]'] = $this->randomMachineName(8);
    $this->drupalGet('crm/contact/add/organization');
    $this->submitForm($edit, 'Save');

    // Check that the Organization has been created.
    $this->assertSession()
      ->pageTextContains('New contact ' . $edit['name[0][value]'] . ' has been created.');

    // Verify that the creation message contains a link to a contact.
    $this->assertSession()
      ->elementExists('xpath', '//div[@data-drupal-messages]//a[contains(@href, "crm/contact/")]');

    // Check that the contact exists in the database.
    $contact = $this->drupalGetContactByName($edit['name[0][value]']);
    $this->assertNotEmpty($contact, 'Contact found in database.');

    // Verify that pages do not show submitted information by default.
    $this->drupalGet('crm/contact/' . $contact->id());
    // $this->assertSession()->pageTextNotContains($contact->getOwner()->getAccountName());
    $this->assertSession()
      ->pageTextContains($this->container->get('date.formatter')
        ->format($contact->get('changed')->value));

  }

  /**
   * Verifies that a transaction rolls back the failed creation.
   */
  public function testFailedPageCreation() {
    // Create a contact.
    $edit = [
      'bundle' => 'organization',
      'name'   => 'testing_transaction_exception',
    ];

    try {
      // An exception is generated by node_test_exception_node_insert() if the
      // title is 'testing_transaction_exception'.
      Contact::create($edit)->save();
      $this->fail('Expected exception has not been thrown.');
    }
    catch (\Exception $e) {
      // Expected exception; just continue testing.
    }

    // Check that the node does not exist in the database.
    $contact = $this->drupalGetContactByName($edit['name']);
    $this->assertFalse($contact);

    // Check that the rollback error was logged.
    $records = static::getWatchdogIdsForTestExceptionRollback();
    // Verify that the rollback explanatory error was logged.
    $this->assertNotEmpty($records);
  }

  /**
   * Creates nodes with different authored dates.
   */
  public function testAuthoredDate() {
    $now = \Drupal::time()->getRequestTime();
    $admin = $this->drupalCreateUser([], NULL, TRUE);
    $this->drupalLogin($admin);

    // Create a node with the default creation date.
    $edit = [
      'name[0][value]' => $this->randomMachineName(8),
    ];
    $this->drupalGet('crm/contact/add/organization');
    $this->submitForm($edit, 'Save');
    $contact = $this->drupalGetContactByName($edit['name[0][value]']);
    $this->assertNotNull($contact->get('created'));

    // Create a node with the custom creation date in the past.
    $date = $now - 86400;
    $edit = [
      'name[0][value]' => $this->randomMachineName(8),
      'created[0][value][date]' => date('Y-m-d', $date),
      'created[0][value][time]' => date('H:i:s', $date),
    ];
    $this->drupalGet('crm/contact/add/organization');
    $this->submitForm($edit, 'Save');
    $contact = $this->drupalGetContactByName($edit['name[0][value]']);
    $this->assertEquals($date, $contact->get('created')->value);

    // Create a node with the custom creation date in the future.
    $date = $now + 86400;
    $edit = [
      'name[0][value]' => $this->randomMachineName(8),
      'created[0][value][date]' => date('Y-m-d', $date),
      'created[0][value][time]' => date('H:i:s', $date),
    ];
    $this->drupalGet('crm/contact/add/organization');
    $this->submitForm($edit, 'Save');
    $contact = $this->drupalGetContactByName($edit['name[0][value]']);
    $this->assertEquals($date, $contact->get('created')->value);

    // Test an invalid date.
    $edit = [
      'name[0][value]' => $this->randomMachineName(8),
      'created[0][value][date]' => '2013-13-13',
      'created[0][value][time]' => '11:00:00',
    ];
    $this->drupalGet('crm/contact/add/organization');
    $this->submitForm($edit, 'Save');
    $this->assertSession()->pageTextContains('The Created on date is invalid.');
    $this->assertFalse($this->drupalGetContactByName($edit['name[0][value]']));

    // Test an invalid time.
    $edit = [
      'name[0][value]' => $this->randomMachineName(8),
      'created[0][value][date]' => '2012-01-01',
      'created[0][value][time]' => '30:00:00',
    ];
    $this->drupalGet('crm/contact/add/organization');
    $this->submitForm($edit, 'Save');
    $this->assertSession()->pageTextContains('The Created on date is invalid.');
    $this->assertFalse($this->drupalGetContactByName($edit['name[0][value]']));
  }

  /**
   * Check crm/contact/add when no contact types exist.
   */
  public function testContactAddWithoutContactTypes() {
    $this->drupalGet('crm/contact/add');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->linkByHrefNotExists('/admin/structure/crm/contact-types/add');

    // Test /crm/contact/add page without content types.
    foreach (\Drupal::entityTypeManager()->getStorage('crm_contact_type')->loadMultiple() as $entity) {
      $entity->delete();
    }

    $this->drupalGet('crm/contact/add');
    $this->assertSession()->statusCodeEquals(403);

    $admin_contact_types = $this->drupalCreateUser([
      'administer crm',
    ]);
    $this->drupalLogin($admin_contact_types);

    $this->drupalGet('crm/contact/add');

    $this->assertSession()->linkByHrefExists('/admin/structure/crm/contact-types/add');
  }

}
