<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Functional\Form;

use Drupal\Tests\BrowserTestBase;

/**
 * Functional tests for the UserContactSettingsForm.
 *
 * @group crm
 * @covers \Drupal\crm\Form\UserContactSettingsForm
 */
class UserContactSettingsFormTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'comment',
    'datetime',
    'name',
    'telephone',
    'address',
  ];

  /**
   * A user with CRM administration permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * A normal user without CRM administration permissions.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $normalUser;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->adminUser = $this->createUser([
      'administer crm',
    ]);

    $this->normalUser = $this->createUser([
      'view any crm_contact',
    ]);
  }

  /**
   * Tests access to the user contact settings form.
   */
  public function testUserContactSettingsFormAccess(): void {
    // Test that anonymous users cannot access the form.
    $this->drupalGet('admin/config/crm/user/settings');
    $this->assertSession()->statusCodeEquals(403);

    // Test that normal users cannot access the form.
    $this->drupalLogin($this->normalUser);
    $this->drupalGet('admin/config/crm/user/settings');
    $this->assertSession()->statusCodeEquals(403);

    // Test that admin users can access the form.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/settings');
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests the form structure and default values.
   */
  public function testFormStructureAndDefaults(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/settings');

    // Verify page title.
    $this->assertSession()->pageTextContains('CRM User settings');

    // Verify form elements are present.
    $this->assertSession()->fieldExists('display_name');
    $this->assertSession()->fieldExists('auto_create_crm_user_contact');
    $this->assertSession()->fieldExists('auto_create_lookup_contact');
    $this->assertSession()->buttonExists('Save configuration');

    // Verify default values (should be unchecked based on default config).
    $this->assertSession()->checkboxNotChecked('display_name');
    $this->assertSession()->checkboxNotChecked('auto_create_crm_user_contact');
    $this->assertSession()->checkboxNotChecked('auto_create_lookup_contact');

    // Verify field labels and descriptions.
    $this->assertSession()->pageTextContains('Display name');
    $this->assertSession()->pageTextContains('Override the User name with the CRM name.');
    $this->assertSession()->pageTextContains('Create event');
    $this->assertSession()->pageTextContains('Create CRM user contact automatically');
    $this->assertSession()->pageTextContains('Automatically create a contact when a user is created.');
    $this->assertSession()->pageTextContains('Lookup contact automatically');
    $this->assertSession()->pageTextContains('Automatically create a lookup contact when a user is created.');
  }

  /**
   * Tests form submission with all checkboxes enabled.
   */
  public function testFormSubmissionWithAllEnabled(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/settings');

    // Enable all checkboxes.
    $edit = [
      'display_name' => TRUE,
      'auto_create_crm_user_contact' => TRUE,
      'auto_create_lookup_contact' => TRUE,
    ];
    $this->submitForm($edit, 'Save configuration');

    // Verify success message.
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Verify that values persist after page reload.
    $this->drupalGet('admin/config/crm/user/settings');
    $this->assertSession()->checkboxChecked('display_name');
    $this->assertSession()->checkboxChecked('auto_create_crm_user_contact');
    $this->assertSession()->checkboxChecked('auto_create_lookup_contact');

    // Verify configuration was actually saved.
    $config = $this->config('crm.user_contact.settings');
    $this->assertTrue($config->get('display_name'));
    $this->assertTrue($config->get('auto_create_crm_user_contact'));
    $this->assertTrue($config->get('auto_create_lookup_contact'));
  }

  /**
   * Tests form submission with all checkboxes disabled.
   */
  public function testFormSubmissionWithAllDisabled(): void {
    // First enable all settings.
    $config = $this->config('crm.user_contact.settings');
    $config->set('display_name', TRUE)
      ->set('auto_create_crm_user_contact', TRUE)
      ->set('auto_create_lookup_contact', TRUE)
      ->save();

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/settings');

    // Verify checkboxes are initially checked.
    $this->assertSession()->checkboxChecked('display_name');
    $this->assertSession()->checkboxChecked('auto_create_crm_user_contact');
    $this->assertSession()->checkboxChecked('auto_create_lookup_contact');

    // Disable all checkboxes.
    $edit = [
      'display_name' => FALSE,
      'auto_create_crm_user_contact' => FALSE,
      'auto_create_lookup_contact' => FALSE,
    ];
    $this->submitForm($edit, 'Save configuration');

    // Verify success message.
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Verify that values persist after page reload.
    $this->drupalGet('admin/config/crm/user/settings');
    $this->assertSession()->checkboxNotChecked('display_name');
    $this->assertSession()->checkboxNotChecked('auto_create_crm_user_contact');
    $this->assertSession()->checkboxNotChecked('auto_create_lookup_contact');

    // Verify configuration was actually saved.
    $config = $this->config('crm.user_contact.settings');
    $this->assertFalse($config->get('display_name'));
    $this->assertFalse($config->get('auto_create_crm_user_contact'));
    $this->assertFalse($config->get('auto_create_lookup_contact'));
  }

  /**
   * Tests form submission with mixed checkbox values.
   */
  public function testFormSubmissionWithMixedValues(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/settings');

    // Set mixed values.
    $edit = [
      'display_name' => TRUE,
      'auto_create_crm_user_contact' => FALSE,
      'auto_create_lookup_contact' => TRUE,
    ];
    $this->submitForm($edit, 'Save configuration');

    // Verify success message.
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Verify that values persist after page reload.
    $this->drupalGet('admin/config/crm/user/settings');
    $this->assertSession()->checkboxChecked('display_name');
    $this->assertSession()->checkboxNotChecked('auto_create_crm_user_contact');
    $this->assertSession()->checkboxChecked('auto_create_lookup_contact');

    // Verify configuration was actually saved.
    $config = $this->config('crm.user_contact.settings');
    $this->assertTrue($config->get('display_name'));
    $this->assertFalse($config->get('auto_create_crm_user_contact'));
    $this->assertTrue($config->get('auto_create_lookup_contact'));
  }

  /**
   * Tests conditional field states functionality.
   */
  public function testConditionalFieldStates(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/settings');

    // Initially, auto_create_lookup_contact should be disabled when
    // auto_create_crm_user_contact is unchecked.
    $this->assertSession()->checkboxNotChecked('auto_create_crm_user_contact');

    // Check that the lookup contact field has the proper states attribute.
    $lookup_field = $this->getSession()->getPage()->findField('auto_create_lookup_contact');
    $this->assertNotNull($lookup_field);

    // Verify that JavaScript states are applied (this tests the #states array).
    $this->assertSession()->elementExists('css', 'input[name="auto_create_lookup_contact"][data-drupal-states]');
  }

  /**
   * Tests form behavior with basic functionality (no JavaScript required).
   */
  public function testFormBasicFunctionality(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/settings');

    // Form should be functional without requiring JavaScript.
    $this->assertSession()->fieldExists('display_name');
    $this->assertSession()->fieldExists('auto_create_crm_user_contact');
    $this->assertSession()->fieldExists('auto_create_lookup_contact');
    $this->assertSession()->buttonExists('Save configuration');

    // Should be able to submit the form successfully.
    $edit = [
      'display_name' => TRUE,
      'auto_create_crm_user_contact' => TRUE,
      'auto_create_lookup_contact' => TRUE,
    ];
    $this->submitForm($edit, 'Save configuration');

    $this->assertSession()->pageTextContains('The configuration options have been saved.');
  }

  /**
   * Tests form validation (should not have validation errors for checkboxes).
   */
  public function testFormValidation(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/settings');

    // Submit form without any changes (should be valid).
    $this->submitForm([], 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Submit form with various combinations (all should be valid).
    $test_cases = [
      [
        'display_name' => TRUE,
        'auto_create_crm_user_contact' => FALSE,
        'auto_create_lookup_contact' => FALSE,
      ],
      [
        'display_name' => FALSE,
        'auto_create_crm_user_contact' => TRUE,
        'auto_create_lookup_contact' => TRUE,
      ],
      [
        'display_name' => TRUE,
        'auto_create_crm_user_contact' => TRUE,
        'auto_create_lookup_contact' => FALSE,
      ],
    ];

    foreach ($test_cases as $edit) {
      $this->drupalGet('admin/config/crm/user/settings');
      $this->submitForm($edit, 'Save configuration');
      $this->assertSession()->pageTextContains('The configuration options have been saved.');
      $this->assertSession()->pageTextNotContains('error');
    }
  }

  /**
   * Tests that form redirects properly after submission.
   */
  public function testFormRedirectAfterSubmission(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/settings');

    $edit = [
      'display_name' => TRUE,
    ];
    $this->submitForm($edit, 'Save configuration');

    // Should stay on the same page after submission.
    $this->assertSession()->addressEquals('admin/config/crm/user/settings');
  }

  /**
   * Tests form behavior with edge case scenarios.
   */
  public function testFormEdgeCases(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/settings');

    // Test that form can handle repeated submissions.
    $edit = [
      'display_name' => TRUE,
      'auto_create_crm_user_contact' => TRUE,
      'auto_create_lookup_contact' => TRUE,
    ];

    // Submit the form multiple times to test idempotency.
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Submit again with different values.
    $edit = [
      'display_name' => FALSE,
      'auto_create_crm_user_contact' => FALSE,
      'auto_create_lookup_contact' => FALSE,
    ];

    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Verify the final configuration state.
    $config = $this->config('crm.user_contact.settings');
    $this->assertFalse($config->get('display_name'));
    $this->assertFalse($config->get('auto_create_crm_user_contact'));
    $this->assertFalse($config->get('auto_create_lookup_contact'));
  }

  /**
   * Tests form accessibility features.
   */
  public function testFormAccessibility(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/settings');

    // Verify that form has proper labels for screen readers.
    $this->assertSession()->elementExists('css', 'label[for="edit-display-name"]');
    $this->assertSession()->elementExists('css', 'label[for="edit-auto-create-crm-user-contact"]');
    $this->assertSession()->elementExists('css', 'label[for="edit-auto-create-lookup-contact"]');

    // Verify that form has proper fieldset structure.
    $this->assertSession()->elementExists('css', 'details[data-drupal-selector="edit-create-event"]');

    // Verify that descriptions exist for fields.
    // (they might be siblings, not adjacent).
    $this->assertSession()->elementExists('css', 'input[name="display_name"]');
    $this->assertSession()->elementExists('css', 'input[name="auto_create_crm_user_contact"]');
    $this->assertSession()->elementExists('css', 'input[name="auto_create_lookup_contact"]');

    // Check that descriptions exist somewhere in the form
    // (more flexible approach).
    $this->assertSession()->pageTextContains('Override the User name with the CRM name.');
    $this->assertSession()->pageTextContains('Automatically create a contact when a user is created.');
    $this->assertSession()->pageTextContains('Automatically create a lookup contact when a user is created.');
  }

  /**
   * Tests form behavior when configuration is pre-populated.
   */
  public function testFormWithPrePopulatedConfiguration(): void {
    // Set initial configuration values.
    $config = $this->config('crm.user_contact.settings');
    $config->set('display_name', TRUE)
      ->set('auto_create_crm_user_contact', TRUE)
      ->set('auto_create_lookup_contact', FALSE)
      ->save();

    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/settings');

    // Verify that form reflects the pre-populated configuration.
    $this->assertSession()->checkboxChecked('display_name');
    $this->assertSession()->checkboxChecked('auto_create_crm_user_contact');
    $this->assertSession()->checkboxNotChecked('auto_create_lookup_contact');

    // Modify some values and submit.
    $edit = [
      'display_name' => FALSE,
      'auto_create_lookup_contact' => TRUE,
    ];
    $this->submitForm($edit, 'Save configuration');

    // Verify success message.
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Verify the updated configuration.
    $this->drupalGet('admin/config/crm/user/settings');
    $this->assertSession()->checkboxNotChecked('display_name');
    $this->assertSession()->checkboxChecked('auto_create_crm_user_contact');
    $this->assertSession()->checkboxChecked('auto_create_lookup_contact');
  }

  /**
   * Tests form submission multiple times to ensure idempotency.
   */
  public function testFormSubmissionIdempotency(): void {
    $this->drupalLogin($this->adminUser);

    $edit = [
      'display_name' => TRUE,
      'auto_create_crm_user_contact' => TRUE,
      'auto_create_lookup_contact' => FALSE,
    ];

    // Submit the same configuration multiple times.
    for ($i = 0; $i < 3; $i++) {
      $this->drupalGet('admin/config/crm/user/settings');
      $this->submitForm($edit, 'Save configuration');
      $this->assertSession()->pageTextContains('The configuration options have been saved.');

      // Verify configuration remains consistent.
      $config = $this->config('crm.user_contact.settings');
      $this->assertTrue($config->get('display_name'));
      $this->assertTrue($config->get('auto_create_crm_user_contact'));
      $this->assertFalse($config->get('auto_create_lookup_contact'));
    }
  }

  /**
   * Tests form behavior with malformed input (should be handled gracefully).
   */
  public function testFormWithMalformedInput(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/settings');

    // Drupal's checkbox handling should convert any truthy value to TRUE
    // and any falsy value to FALSE, so malformed input should be
    // handled gracefully.
    $edit = [
      'display_name' => 'invalid_value',
      'auto_create_crm_user_contact' => '1',
      'auto_create_lookup_contact' => '0',
    ];
    $this->submitForm($edit, 'Save configuration');

    // Should still save successfully with boolean conversion.
    $this->assertSession()->pageTextContains('The configuration options have been saved.');

    // Verify that values were properly converted to booleans.
    $config = $this->config('crm.user_contact.settings');
    // 'invalid_value' is truthy
    $this->assertTrue($config->get('display_name'));
    // '1' is truthy
    $this->assertTrue($config->get('auto_create_crm_user_contact'));
    // '0' is falsy
    $this->assertFalse($config->get('auto_create_lookup_contact'));
  }

  /**
   * Tests form navigation and breadcrumbs.
   */
  public function testFormNavigationAndBreadcrumbs(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/settings');

    // Verify that we're in the correct administrative section.
    $this->assertSession()->addressEquals('admin/config/crm/user/settings');

    // Verify that the form is accessible through the admin interface.
    $this->assertSession()->statusCodeEquals(200);

    // Test that form submission doesn't break navigation.
    $edit = ['display_name' => TRUE];
    $this->submitForm($edit, 'Save configuration');

    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->addressEquals('admin/config/crm/user/settings');
  }

  /**
   * Tests form with different user roles and permissions.
   */
  public function testFormWithDifferentPermissions(): void {
    // Create a user with partial permissions.
    $partial_user = $this->createUser([
      'view any crm_contact',
      'edit any crm_contact',
    ]);

    // Test that user without 'administer crm' permission cannot access.
    $this->drupalLogin($partial_user);
    $this->drupalGet('admin/config/crm/user/settings');
    $this->assertSession()->statusCodeEquals(403);

    // Test that user with 'administer crm' permission can access.
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/settings');
    $this->assertSession()->statusCodeEquals(200);

    // Verify form functionality works for admin user.
    $edit = ['display_name' => TRUE];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved.');
  }

  /**
   * Tests form behavior when configuration becomes corrupted or missing.
   */
  public function testFormWithMissingConfiguration(): void {
    $this->drupalLogin($this->adminUser);

    // The form should handle missing configuration
    // gracefully by using defaults.
    $this->drupalGet('admin/config/crm/user/settings');
    $this->assertSession()->statusCodeEquals(200);

    // Form should still be functional and use default values.
    $this->assertSession()->checkboxNotChecked('display_name');
    $this->assertSession()->checkboxNotChecked('auto_create_crm_user_contact');
    $this->assertSession()->checkboxNotChecked('auto_create_lookup_contact');

    // Should be able to save new configuration.
    $edit = [
      'display_name' => TRUE,
      'auto_create_crm_user_contact' => TRUE,
      'auto_create_lookup_contact' => TRUE,
    ];
    $this->submitForm($edit, 'Save configuration');
    $this->assertSession()->pageTextContains('The configuration options have been saved.');
  }

  /**
   * Tests the details element expand/collapse functionality.
   */
  public function testDetailsElementFunctionality(): void {
    $this->drupalLogin($this->adminUser);
    $this->drupalGet('admin/config/crm/user/settings');

    // Verify that the details element is present and open by default.
    $details_element = $this->getSession()->getPage()->find('css', 'details[data-drupal-selector="edit-create-event"]');
    $this->assertNotNull($details_element);
    $this->assertTrue($details_element->hasAttribute('open'));

    // Verify that the summary text is correct.
    $summary = $details_element->find('css', 'summary');
    $this->assertNotNull($summary);
    $this->assertEquals('Create event', $summary->getText());

    // Verify that the nested checkboxes are inside the details element.
    $auto_create_checkbox = $details_element->find('css', 'input[name="auto_create_crm_user_contact"]');
    $lookup_checkbox = $details_element->find('css', 'input[name="auto_create_lookup_contact"]');
    $this->assertNotNull($auto_create_checkbox);
    $this->assertNotNull($lookup_checkbox);
  }

  /**
   * Tests form with concurrent access from multiple sessions.
   */
  public function testFormConcurrentAccess(): void {
    $this->drupalLogin($this->adminUser);

    // Simulate first session setting configuration.
    $this->drupalGet('admin/config/crm/user/settings');
    $edit1 = [
      'display_name' => TRUE,
      'auto_create_crm_user_contact' => FALSE,
    ];
    $this->submitForm($edit1, 'Save configuration');

    // Verify first submission worked.
    $config = $this->config('crm.user_contact.settings');
    $this->assertTrue($config->get('display_name'));
    $this->assertFalse($config->get('auto_create_crm_user_contact'));

    // Simulate second session (different configuration).
    $this->drupalGet('admin/config/crm/user/settings');
    $edit2 = [
      'display_name' => FALSE,
      'auto_create_crm_user_contact' => TRUE,
      'auto_create_lookup_contact' => TRUE,
    ];
    $this->submitForm($edit2, 'Save configuration');

    // Verify second submission overwrote the first.
    $config = $this->config('crm.user_contact.settings');
    $this->assertFalse($config->get('display_name'));
    $this->assertTrue($config->get('auto_create_crm_user_contact'));
    $this->assertTrue($config->get('auto_create_lookup_contact'));
  }

}
