<?php

namespace Drupal\crm;

use Drupal\Core\Config\Entity\ConfigEntityListBuilder;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityStorageInterface;
use Drupal\Core\Entity\EntityTypeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a listing of Detail types.
 */
class DetailTypeListBuilder extends ConfigEntityListBuilder {

  /**
   * Contact Detail Type storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $contactDetailTypeStorage;

  /**
   * {@inheritdoc}
   */
  public static function createInstance(ContainerInterface $container, EntityTypeInterface $entity_type) {
    return new static(
      $entity_type,
      $container->get('entity_type.manager')->getStorage($entity_type->id()),
      $container->get('entity_type.manager')->getStorage('crm_contact_detail_type'),
    );
  }

  /**
   * Constructs a new EntityListBuilder object.
   *
   * @param \Drupal\Core\Entity\EntityTypeInterface $entity_type
   *   The entity type definition.
   * @param \Drupal\Core\Entity\EntityStorageInterface $storage
   *   The entity storage class.
   * @param \Drupal\Core\Entity\EntityStorageInterface $contact_detail_type_storage
   *   The entity storage class.
   */
  public function __construct(EntityTypeInterface $entity_type, EntityStorageInterface $storage, EntityStorageInterface $contact_detail_type_storage) {
    parent::__construct($entity_type, $storage);

    $this->contactDetailTypeStorage = $contact_detail_type_storage;
  }

  /**
   * {@inheritdoc}
   */
  public function buildHeader() {
    $header['label'] = $this->t('Label');
    $header['id'] = $this->t('Machine name');
    $header['status'] = $this->t('Status');
    $header['bundles'] = $this->t('Bundles');

    return $header + parent::buildHeader();
  }

  /**
   * {@inheritdoc}
   */
  public function buildRow(EntityInterface $entity) {

    $bundle_ids = $entity->get('bundles');
    $contact_detail_types = $this->contactDetailTypeStorage->loadMultiple($bundle_ids) ?? [];
    $bundles = array_map(
      fn($contact_detail_type): string => $contact_detail_type->label(),
       $contact_detail_types);

    if (empty($bundles)) {
      $bundle = $this->t('Any');
    }
    elseif ($entity->get('negate')) {
      $bundle = $this->t('Except: @bundles', ['@bundles' => implode(',', $bundles)]);
    }
    else {
      $bundle = $this->t('Only: @bundles', ['@bundles' => implode(',', $bundles)]);
    }

    /** @var \Drupal\crm\CrmDetailTypeInterface $entity */
    $row['label'] = $entity->label();
    $row['id'] = $entity->id();
    $row['status'] = $entity->status() ? $this->t('Enabled') : $this->t('Disabled');
    $row['bundles'] = $bundle;

    return $row + parent::buildRow($entity);
  }

}
