<?php

namespace Drupal\crm\Entity;

use Drupal\Core\Config\Entity\ConfigEntityBundleBase;
use Drupal\Core\Entity\Attribute\ConfigEntityType;
use Drupal\Core\Entity\EntityDeleteForm;
use Drupal\Core\Entity\Routing\AdminHtmlRouteProvider;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\crm\ContactTypeAccessControlHandler;
use Drupal\crm\ContactTypeListBuilder;
use Drupal\crm\CrmContactTypeInterface;
use Drupal\crm\Form\ContactTypeForm;

/**
 * Defines the Contact type configuration entity.
 */
#[ConfigEntityType(
  id: 'crm_contact_type',
  label: new TranslatableMarkup('Contact type'),
  label_collection: new TranslatableMarkup('Contact types'),
  label_singular: new TranslatableMarkup('contact type'),
  label_plural: new TranslatableMarkup('contacts types'),
  label_count: [
    'singular' => '@count contacts type',
    'plural' => '@count contacts types',
  ],
  handlers: [
    'access' => ContactTypeAccessControlHandler::class,
    'form' => [
      'add' => ContactTypeForm::class,
      'edit' => ContactTypeForm::class,
      'delete' => EntityDeleteForm::class,
    ],
    'list_builder' => ContactTypeListBuilder::class,
    'route_provider' => [
      'html' => AdminHtmlRouteProvider::class,
    ],
  ],
  admin_permission: 'administer crm',
  bundle_of: 'crm_contact',
  config_prefix: 'crm_contact_type',
  entity_keys: [
    'id' => 'id',
    'label' => 'label',
    'uuid' => 'uuid',
  ],
  links: [
    'add-form' => '/admin/structure/crm/contact-types/add',
    'edit-form' => '/admin/structure/crm/contact-types/manage/{crm_contact_type}',
    'delete-form' => '/admin/structure/crm/contact-types/manage/{crm_contact_type}/delete',
    'collection' => '/admin/structure/crm/contact-types',
  ],
  config_export: [
    'id',
    'label',
    'description',
    'help',
    'new_revision',
    'uuid',
    'locked',
    'date',
  ],
)]
class ContactType extends ConfigEntityBundleBase implements CrmContactTypeInterface {

  /**
   * The machine name of this contact type.
   *
   * @var string
   */
  protected $id;

  /**
   * The human-readable name of the contact type.
   *
   * @var string
   */
  protected $label;

  /**
   * A description of the contact type.
   *
   * @var string|null
   */
  protected $description;

  /**
   * Whether the contact type is locked.
   *
   * @var bool
   */
  protected $locked = FALSE;

  /**
   * Help text for the contact type.
   *
   * @var string
   */
  protected $help;

  /**
   * Whether a new revision should be created by default.
   *
   * @var bool
   */
  protected $new_revision = TRUE;

  /**
   * {@inheritdoc}
   */
  public function id() {
    return $this->id;
  }

  /**
   * Is the contact type locked?
   */
  public function isLocked() {
    return $this->locked;
  }

  /**
   * {@inheritdoc}
   */
  public function getHelp() {
    return $this->help ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function getDescription() {
    return $this->description ?? '';
  }

  /**
   * {@inheritdoc}
   */
  public function setDescription($description) {
    $this->description = $description;

    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function shouldCreateNewRevision() {
    return $this->new_revision;
  }

  /**
   * {@inheritdoc}
   */
  public function setNewRevision($new_revision) {
    $this->new_revision = $new_revision;
  }

}
