<?php

namespace Drupal\crm\EventSubscriber;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\crm\Event\CrmUserContactEvent;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;

/**
 * Subscribe to crm user contact create events.
 */
class DefaultCrmUserContactSubscriber implements EventSubscriberInterface {

  /**
   * The crm contact storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $contactStorage;

  /**
   * The crm contact detail storage.
   *
   * @var \Drupal\Core\Entity\EntityStorageInterface
   */
  protected $contactDetailStorage;

  /**
   * The crm user contact settings.
   *
   * @var \Drupal\Core\Config\ImmutableConfig
   */
  protected $settings;

  /**
   * Constructs a new DefaultCrmUserContactSubscriber object.
   *
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  public function __construct(EntityTypeManagerInterface $entity_type_manager, ConfigFactoryInterface $config_factory) {
    $this->settings = $config_factory->get('crm.user_contact.settings');
    $this->contactStorage = $entity_type_manager->getStorage('crm_contact');
    $this->contactDetailStorage = $entity_type_manager->getStorage('crm_contact_detail');
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents() {
    $events = [];
    $events[CrmUserContactEvent::EVENT_NAME] = ['onCrmUserContactCreate', 100];

    return $events;
  }

  /**
   * React to a crm user contact create event.
   *
   * @param \Drupal\crm\Event\CrmUserContactEvent $event
   *   The crm user contact event.
   */
  public function onCrmUserContactCreate(CrmUserContactEvent $event) {

    $crm_user_contact = $event->getCrmUserContact();
    $contact = $crm_user_contact->getContact();
    $lookup = $this->settings->get('auto_create_lookup_contact');
    $user = $crm_user_contact->getUser();
    $user_email = $user->getEmail();
    if ($lookup && $contact == NULL) {
      $contact_detail_emails = $this->contactDetailStorage->getQuery()
        ->condition('email', $user_email)
        ->condition('bundle', 'email')
        ->sort('status', 'DESC')
        ->accessCheck(FALSE)
        ->execute();

      $contact_ids = $this->contactStorage->getQuery()
        ->condition('emails', $contact_detail_emails, 'IN')
        ->condition('bundle', 'person')
        ->sort('status', 'DESC')
        ->accessCheck(FALSE)
        ->execute();

      if ($contact == NULL && count($contact_ids) == 1) {
        $contact = $this->contactStorage->load(reset($contact_ids));
      }

    }
    if ($contact == NULL) {
      $contact = $this->contactStorage
        ->create(['bundle' => 'person']);

      $email = $this->contactDetailStorage
        ->create([
          'bundle' => 'email',
          'email' => $user_email,
          'type' => 'main',
        ]);

      $email->save();
      $contact->set('emails', $email);
      $contact->set('full_name', ['given' => $user->getDisplayName()]);

      $contact->save();
    }
    $crm_user_contact->setContact($contact);
    $event->setCrmUserContact($crm_user_contact);
  }

}
