<?php

declare(strict_types=1);

namespace Drupal\crm\Hook;

use Drupal\Core\Hook\Attribute\Hook;
use Drupal\search\SearchIndexInterface;

/**
 * Hooks relating to contacts.
 */
class SearchHooks {

  public function __construct(
    protected ?SearchIndexInterface $searchIndex = NULL,
  ) {}

  /**
   * Implements hook_ENTITY_TYPE_insert().
   */
  #[Hook('crm_contact_insert')]
  public function crmContactInsert($contact): void {
    $this->reindexContact($contact->id());
  }

  /**
   * Implements hook_ENTITY_TYPE_update().
   */
  #[Hook('crm_contact_update')]
  public function crmContactUpdate($contact): void {
    $this->reindexContact($contact->id());
  }

  /**
   * Implements hook_ENTITY_TYPE_delete().
   */
  #[Hook('crm_contact_delete')]
  public function crmContactDelete($contact) {
    // Mark for removal from search index.
    $this->reindexContact($contact->id());
  }

  /**
   * Implements hook_ENTITY_TYPE_insert() for crm_contact_detail.
   */
  #[Hook('crm_contact_detail_insert')]
  public function crmContactDetailInsert($contact_detail): void {
    $this->reindexContactFromDetail($contact_detail);
  }

  /**
   * Implements hook_ENTITY_TYPE_update() for crm_contact_detail.
   */
  #[Hook('crm_contact_detail_update')]
  public function crmContactDetailUpdate($contact_detail): void {
    $this->reindexContactFromDetail($contact_detail);
  }

  /**
   * Implements hook_ENTITY_TYPE_delete() for crm_contact_detail.
   */
  #[Hook('crm_contact_detail_delete')]
  public function crmContactDetailDelete($contact_detail): void {
    $this->reindexContactFromDetail($contact_detail);
  }

  /**
   * Marks a contact for reindexing by search module.
   *
   * @param int|string $contact_id
   *   The contact ID.
   */
  protected function reindexContact(int|string $contact_id): void {
    if ($this->searchIndex !== NULL) {
      // Reindex contact indexed by the CRM contact search plugin.
      $this->searchIndex->markForReindex('crm_contact_search', (int) $contact_id);
    }
  }

  /**
   * Marks a contact for reindexing from a contact detail entity.
   *
   * @param \Drupal\crm\CrmContactDetailInterface $contact_detail
   *   The contact detail entity.
   */
  protected function reindexContactFromDetail($contact_detail): void {
    if (!$contact_detail->get('crm_contact')->isEmpty()) {
      $contact_id = $contact_detail->get('crm_contact')->target_id;
      if ($contact_id) {
        $this->reindexContact($contact_id);
      }
    }
  }

}
