<?php

declare(strict_types=1);

namespace Drupal\crm\Plugin\Search;

use Drupal\Core\Access\AccessibleInterface;
use Drupal\Core\Access\AccessResult;
use Drupal\Core\Config\Config;
use Drupal\Core\Database\Connection;
use Drupal\Core\Database\Query\PagerSelectExtender;
use Drupal\Core\Database\StatementInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\crm\CrmContactInterface;
use Drupal\search\Attribute\Search;
use Drupal\search\Plugin\ConfigurableSearchPluginBase;
use Drupal\search\Plugin\SearchIndexingInterface;
use Drupal\search\SearchIndexInterface;
use Drupal\search\SearchQuery;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Handles searching for contact entities using the Search module index.
 */
#[Search(
  id: 'crm_contact_search',
  title: new TranslatableMarkup('Contacts'),
)]
class ContactSearch extends ConfigurableSearchPluginBase implements AccessibleInterface, SearchIndexingInterface {

  /**
   * The current database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $database;

  /**
   * The replica database connection.
   *
   * @var \Drupal\Core\Database\Connection
   */
  protected Connection $databaseReplica;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected ModuleHandlerInterface $moduleHandler;

  /**
   * A config object for 'search.settings'.
   *
   * @var \Drupal\Core\Config\Config
   */
  protected Config $searchSettings;

  /**
   * The current user.
   *
   * @var \Drupal\Core\Session\AccountInterface
   */
  protected AccountInterface $currentUser;

  /**
   * The Renderer service to format the contact.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected RendererInterface $renderer;

  /**
   * The search index.
   *
   * @var \Drupal\search\SearchIndexInterface
   */
  protected SearchIndexInterface $searchIndex;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('database'),
      $container->get('entity_type.manager'),
      $container->get('module_handler'),
      $container->get('config.factory')->get('search.settings'),
      $container->get('renderer'),
      $container->get('current_user'),
      $container->get('database.replica'),
      $container->get('search.index')
    );
  }

  /**
   * Constructs a ContactSearch object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Database\Connection $database
   *   The current database connection.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler.
   * @param \Drupal\Core\Config\Config $search_settings
   *   A config object for 'search.settings'.
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer.
   * @param \Drupal\Core\Session\AccountInterface $current_user
   *   The current user.
   * @param \Drupal\Core\Database\Connection $database_replica
   *   The replica database connection.
   * @param \Drupal\search\SearchIndexInterface $search_index
   *   The search index.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    Connection $database,
    EntityTypeManagerInterface $entity_type_manager,
    ModuleHandlerInterface $module_handler,
    Config $search_settings,
    RendererInterface $renderer,
    AccountInterface $current_user,
    Connection $database_replica,
    SearchIndexInterface $search_index,
  ) {
    $this->database          = $database;
    $this->databaseReplica   = $database_replica;
    $this->entityTypeManager = $entity_type_manager;
    $this->moduleHandler     = $module_handler;
    $this->searchSettings    = $search_settings;
    $this->renderer          = $renderer;
    $this->currentUser       = $current_user;
    $this->searchIndex       = $search_index;

    parent::__construct($configuration, $plugin_id, $plugin_definition);

    $this->addCacheTags(['crm_contact_list']);
  }

  /**
   * {@inheritdoc}
   */
  public function access($operation = 'view', ?AccountInterface $account = NULL, $return_as_object = FALSE) {
    // Allow access if user can view contacts OR has admin permission.
    $result = AccessResult::allowedIfHasPermission($account, 'view any crm_contact')
      ->orIf(AccessResult::allowedIfHasPermission($account, 'administer crm'));
    return $return_as_object ? $result : $result->isAllowed();
  }

  /**
   * {@inheritdoc}
   */
  public function getType() {
    return $this->getPluginId();
  }

  /**
   * {@inheritdoc}
   */
  public function execute() {
    if ($this->isSearchExecutable()) {
      $results = $this->findResults();

      if ($results) {
        return $this->prepareResults($results);
      }
    }

    return [];
  }

  /**
   * Queries to find search results, and sets status messages.
   *
   * This method can assume that $this->isSearchExecutable() has already been
   * checked and returned TRUE.
   *
   * @return \Drupal\Core\Database\StatementInterface|null
   *   Results from search query execute() method, or NULL if the search
   *   failed.
   */
  protected function findResults(): ?StatementInterface {
    $keys = $this->keywords;

    // Build matching conditions.
    $query = $this->databaseReplica
      ->select('search_index', 'i')
      ->extend(SearchQuery::class)
      ->extend(PagerSelectExtender::class);
    $query->join('crm_contact', 'c', '[c].[id] = [i].[sid]');
    $query->searchExpression($keys, $this->getPluginId());

    // Only search published contacts for non-admins.
    if (!$this->currentUser->hasPermission('administer crm')) {
      $query->condition('c.status', 1);
    }

    // Run the query. Need to add fields and groupBy for SearchQuery to work.
    // Also explicitly select sid for loading entities later.
    $find = $query
      ->fields('i', ['sid', 'langcode'])
      ->groupBy('i.langcode')
      ->groupBy('i.sid')
      ->limit(10)
      ->execute();

    // Check query status and set messages if needed.
    $status = $query->getStatus();

    if ($status & SearchQuery::NO_POSITIVE_KEYWORDS) {
      return NULL;
    }

    return $find;
  }

  /**
   * Prepares search results for rendering.
   *
   * @param \Drupal\Core\Database\StatementInterface $found
   *   Results found from a successful search query execute() method.
   *
   * @return array
   *   Array of search result item render arrays (empty array if no results).
   */
  protected function prepareResults(StatementInterface $found): array {
    $results = [];

    $contact_storage = $this->entityTypeManager->getStorage('crm_contact');
    $keys            = $this->keywords;

    foreach ($found as $item) {
      /** @var \Drupal\crm\CrmContactInterface $contact */
      $contact = $contact_storage->load($item->sid);

      if (!$contact) {
        continue;
      }

      // Render the contact for snippet generation.
      $contact_render = $this->entityTypeManager->getViewBuilder('crm_contact');
      $build          = $contact_render->view($contact, 'search_result');
      unset($build['#theme']);

      $rendered = $this->renderer->renderInIsolation($build);
      $this->addCacheableDependency($contact);

      /** @var \Drupal\crm\CrmContactTypeInterface $type */
      $type = $this->entityTypeManager->getStorage('crm_contact_type')->load($contact->bundle());

      $result = [
        'link'    => $contact->toUrl('canonical', ['absolute' => TRUE])->toString(),
        'type'    => $type->label(),
        'title'   => $contact->label(),
        'contact' => $contact,
        'score'   => $item->calculated_score,
        'snippet' => search_excerpt($keys, $rendered, $item->langcode ?? 'en'),
      ];

      $results[] = $result;
    }

    return $results;
  }

  /**
   * {@inheritdoc}
   */
  public function updateIndex(): void {
    // Interpret the cron limit setting as the maximum number of contacts to
    // index per cron run.
    $limit = (int) $this->searchSettings->get('index.cron_limit');

    $query = $this->databaseReplica->select('crm_contact', 'c');
    $query->addField('c', 'id');
    $query->leftJoin('search_dataset', 'sd', '[sd].[sid] = [c].[id] AND [sd].[type] = :type', [':type' => $this->getPluginId()]);
    $query->addExpression('CASE MAX([sd].[reindex]) WHEN NULL THEN 0 ELSE 1 END', 'ex');
    $query->addExpression('MAX([sd].[reindex])', 'ex2');
    $query->condition(
      $query->orConditionGroup()
        ->where('[sd].[sid] IS NULL')
        ->condition('sd.reindex', 0, '<>')
    );
    $query->orderBy('ex', 'DESC')
      ->orderBy('ex2')
      ->orderBy('c.id')
      ->groupBy('c.id')
      ->range(0, $limit);

    $ids = $query->execute()->fetchCol();
    if (!$ids) {
      return;
    }

    $contact_storage = $this->entityTypeManager->getStorage('crm_contact');
    $words           = [];
    try {
      foreach ($contact_storage->loadMultiple($ids) as $contact) {
        $words += $this->indexContact($contact);
      }
    }
    finally {
      $this->searchIndex->updateWordWeights($words);
    }
  }

  /**
   * Indexes a single contact.
   *
   * @param \Drupal\crm\CrmContactInterface $contact
   *   The contact to index.
   *
   * @return array
   *   An array of words to update after indexing.
   */
  protected function indexContact(CrmContactInterface $contact): array {
    $words = [];

    // Build text content for indexing.
    // Start with the contact label/name as highest priority.
    $text = '<h1>' . htmlspecialchars($contact->label() ?? '') . '</h1>';

    // Try to render the contact entity for additional content.
    try {
      $contact_render = $this->entityTypeManager->getViewBuilder('crm_contact');
      $build          = $contact_render->view($contact, 'search_index');
      unset($build['#theme']);
      $rendered = $this->renderer->renderInIsolation($build);
      $text .= ' ' . $rendered;
    }
    catch (\Exception $e) {
      // If rendering fails, just use the label.
    }

    // Update index, using search index "type" equal to the plugin ID.
    $words += $this->searchIndex->index($this->getPluginId(), $contact->id(), 'en', $text, FALSE);

    return $words;
  }

  /**
   * {@inheritdoc}
   */
  public function indexClear(): void {
    // All ContactSearch pages share a common search index "type" equal to
    // the plugin ID.
    $this->searchIndex->clear($this->getPluginId());
  }

  /**
   * {@inheritdoc}
   */
  public function markForReindex(): void {
    // All ContactSearch pages share a common search index "type" equal to
    // the plugin ID.
    $this->searchIndex->markForReindex($this->getPluginId());
  }

  /**
   * {@inheritdoc}
   */
  public function indexStatus(): array {
    $total     = $this->database->query('SELECT COUNT(*) FROM {crm_contact}')->fetchField();
    $remaining = $this->database->query(
      "SELECT COUNT(DISTINCT [c].[id]) FROM {crm_contact} [c] LEFT JOIN {search_dataset} [sd] ON [sd].[sid] = [c].[id] AND [sd].[type] = :type WHERE [sd].[sid] IS NULL OR [sd].[reindex] <> 0",
      [':type' => $this->getPluginId()]
    )->fetchField();

    return ['remaining' => $remaining, 'total' => $total];
  }

  /**
   * {@inheritdoc}
   */
  public function buildConfigurationForm(array $form, FormStateInterface $form_state): array {
    // Contact search currently has no configuration options.
    // This method can be extended in the future to add options like:
    // - Ranking factors for different fields
    // - Search result display options
    // - Filter options by contact type.
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitConfigurationForm(array &$form, FormStateInterface $form_state): void {
    // No configuration to save currently.
    // This will be implemented when configuration options are added.
  }

  /**
   * {@inheritdoc}
   */
  public function getHelp(): array {
    $help = [
      'list' => [
        '#theme' => 'item_list',
        '#items' => [
          $this->t('Contact search looks for contact names, email addresses, telephone numbers, and addresses. Example: john@example.com would match contacts with that email address.'),
          $this->t('You can use * as a wildcard within your keyword. Example: john* would match john, johnson, and johnathon.'),
          $this->t('The search will find matches in contact names as well as their associated contact details (emails, telephones, and addresses).'),
        ],
      ],
    ];

    return $help;
  }

}
