<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel\Form;

use Drupal\Core\Form\FormState;
use Drupal\crm\Form\UserContactSettingsForm;
use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;

/**
 * Kernel tests for the UserContactSettingsForm.
 *
 * @group crm
 * @covers \Drupal\crm\Form\UserContactSettingsForm
 */
class UserContactSettingsFormTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'comment',
    'datetime',
    'inline_entity_form',
    'primary_entity_reference',
    'name',
    'telephone',
    'address',
  ];

  /**
   * The form under test.
   *
   * @var \Drupal\crm\Form\UserContactSettingsForm
   */
  protected $form;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installConfig(['crm']);

    $this->configFactory = $this->container->get('config.factory');
    $this->form = UserContactSettingsForm::create($this->container);
  }

  /**
   * Tests form creation and dependency injection.
   */
  public function testFormCreation(): void {
    $this->assertInstanceOf(UserContactSettingsForm::class, $this->form);
  }

  /**
   * Tests building the form with default configuration values.
   */
  public function testBuildFormWithDefaultConfiguration(): void {
    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    $this->assertIsArray($form);

    // Test that form includes all expected elements.
    $this->assertArrayHasKey('display_name', $form);
    $this->assertArrayHasKey('create_event', $form);
    $this->assertArrayHasKey('auto_create_crm_user_contact', $form['create_event']);
    $this->assertArrayHasKey('auto_create_lookup_contact', $form['create_event']);

    // Test default values from configuration.
    $this->assertFalse($form['display_name']['#default_value']);
    $this->assertFalse($form['create_event']['auto_create_crm_user_contact']['#default_value']);
    $this->assertFalse($form['create_event']['auto_create_lookup_contact']['#default_value']);

    // Test form structure.
    $this->assertEquals('checkbox', $form['display_name']['#type']);
    $this->assertEquals('details', $form['create_event']['#type']);
    $this->assertTrue($form['create_event']['#open']);
    $this->assertEquals('checkbox', $form['create_event']['auto_create_crm_user_contact']['#type']);
    $this->assertEquals('checkbox', $form['create_event']['auto_create_lookup_contact']['#type']);

    // Test conditional states.
    $expected_states = [
      'enabled' => [
        ':input[name="auto_create_crm_user_contact"]' => ['checked' => TRUE],
      ],
    ];
    $this->assertEquals($expected_states, $form['create_event']['auto_create_lookup_contact']['#states']);
  }

  /**
   * Tests building the form with custom configuration values.
   */
  public function testBuildFormWithCustomConfiguration(): void {
    // Set custom configuration values.
    $config = $this->configFactory->getEditable('crm.user_contact.settings');
    $config->set('display_name', TRUE)
      ->set('auto_create_crm_user_contact', TRUE)
      ->set('auto_create_lookup_contact', TRUE)
      ->save();

    $form_state = new FormState();
    $form = $this->form->buildForm([], $form_state);

    // Test that custom values are reflected in form defaults.
    $this->assertTrue($form['display_name']['#default_value']);
    $this->assertTrue($form['create_event']['auto_create_crm_user_contact']['#default_value']);
    $this->assertTrue($form['create_event']['auto_create_lookup_contact']['#default_value']);
  }

  /**
   * Tests form submission and configuration saving.
   */
  public function testFormSubmissionSavesConfiguration(): void {
    $form_state = new FormState();
    $form_state->setValues([
      'display_name' => TRUE,
      'auto_create_crm_user_contact' => TRUE,
      'auto_create_lookup_contact' => FALSE,
    ]);

    $form = $this->form->buildForm([], $form_state);
    $this->form->submitForm($form, $form_state);

    // Verify configuration was saved.
    $config = $this->configFactory->get('crm.user_contact.settings');
    $this->assertTrue($config->get('display_name'));
    $this->assertTrue($config->get('auto_create_crm_user_contact'));
    $this->assertFalse($config->get('auto_create_lookup_contact'));
  }

  /**
   * Tests form submission with all false values.
   */
  public function testFormSubmissionWithAllFalseValues(): void {
    // First set some true values.
    $config = $this->configFactory->getEditable('crm.user_contact.settings');
    $config->set('display_name', TRUE)
      ->set('auto_create_crm_user_contact', TRUE)
      ->set('auto_create_lookup_contact', TRUE)
      ->save();

    $form_state = new FormState();
    $form_state->setValues([
      'display_name' => FALSE,
      'auto_create_crm_user_contact' => FALSE,
      'auto_create_lookup_contact' => FALSE,
    ]);

    $form = $this->form->buildForm([], $form_state);
    $this->form->submitForm($form, $form_state);

    // Verify all values were set to false.
    $config = $this->configFactory->get('crm.user_contact.settings');
    $this->assertFalse($config->get('display_name'));
    $this->assertFalse($config->get('auto_create_crm_user_contact'));
    $this->assertFalse($config->get('auto_create_lookup_contact'));
  }

  /**
   * Tests form submission with mixed values.
   */
  public function testFormSubmissionWithMixedValues(): void {
    $form_state = new FormState();
    $form_state->setValues([
      'display_name' => FALSE,
      'auto_create_crm_user_contact' => TRUE,
      'auto_create_lookup_contact' => TRUE,
    ]);

    $form = $this->form->buildForm([], $form_state);
    $this->form->submitForm($form, $form_state);

    // Verify mixed values were saved correctly.
    $config = $this->configFactory->get('crm.user_contact.settings');
    $this->assertFalse($config->get('display_name'));
    $this->assertTrue($config->get('auto_create_crm_user_contact'));
    $this->assertTrue($config->get('auto_create_lookup_contact'));
  }

  /**
   * Tests that configuration persists across form instances.
   */
  public function testConfigurationPersistence(): void {
    // Submit form with specific values.
    $form_state = new FormState();
    $form_state->setValues([
      'display_name' => TRUE,
      'auto_create_crm_user_contact' => FALSE,
      'auto_create_lookup_contact' => TRUE,
    ]);

    $form = $this->form->buildForm([], $form_state);
    $this->form->submitForm($form, $form_state);

    // Create a new form instance and verify it loads the saved configuration.
    $new_form = UserContactSettingsForm::create($this->container);

    $new_form_state = new FormState();
    $built_form = $new_form->buildForm([], $new_form_state);

    $this->assertTrue($built_form['display_name']['#default_value']);
    $this->assertFalse($built_form['create_event']['auto_create_crm_user_contact']['#default_value']);
    $this->assertTrue($built_form['create_event']['auto_create_lookup_contact']['#default_value']);
  }

  /**
   * Tests form validation (should pass as this is a simple config form).
   */
  public function testFormValidation(): void {
    $form_state = new FormState();
    $form_state->setValues([
      'display_name' => TRUE,
      'auto_create_crm_user_contact' => TRUE,
      'auto_create_lookup_contact' => TRUE,
    ]);

    $form = $this->form->buildForm([], $form_state);

    // The form should validate successfully with any boolean values.
    $this->form->validateForm($form, $form_state);

    // No validation errors should be set.
    $this->assertEmpty($form_state->getErrors());
  }

  /**
   * Tests form with edge case values (checkbox unchecked scenarios).
   */
  public function testFormWithEdgeCaseValues(): void {
    $form_state = new FormState();
    $form_state->setValues([
      'display_name' => 0,
      'auto_create_crm_user_contact' => 0,
      'auto_create_lookup_contact' => 0,
    ]);

    $form = $this->form->buildForm([], $form_state);
    $this->form->submitForm($form, $form_state);

    // Verify that unchecked checkbox values (0) are stored correctly.
    $config = $this->configFactory->get('crm.user_contact.settings');
    $this->assertFalse((bool) $config->get('display_name'));
    $this->assertFalse((bool) $config->get('auto_create_crm_user_contact'));
    $this->assertFalse((bool) $config->get('auto_create_lookup_contact'));
  }

}
