<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel\Plugin\Validation\Constraint;

use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\ContactType;
use Drupal\crm\Entity\Relationship;
use Drupal\crm\Entity\RelationshipType;

/**
 * Tests the RelationshipContacts constraint validator.
 *
 * @group crm
 */
class RelationshipContactsConstraintValidatorTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'address',
    'comment',
    'crm',
    'inline_entity_form',
    'primary_entity_reference',
    'datetime',
    'name',
    'telephone',
  ];

  /**
   * Contact type: Person.
   *
   * @var \Drupal\crm\Entity\ContactType
   */
  protected ContactType $personType;

  /**
   * Contact type: Organization.
   *
   * @var \Drupal\crm\Entity\ContactType
   */
  protected ContactType $organizationType;

  /**
   * Relationship type: Employment (Person to Organization).
   *
   * @var \Drupal\crm\Entity\RelationshipType
   */
  protected RelationshipType $employmentType;

  /**
   * Relationship type: Friendship (Person to Person).
   *
   * @var \Drupal\crm\Entity\RelationshipType
   */
  protected RelationshipType $friendshipType;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('comment');
    $this->installSchema('comment', ['comment_entity_statistics']);
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_relationship');

    $this->installConfig(['crm', 'name']);

    // Create relationship types.
    $this->employmentType = RelationshipType::create([
      'id'             => 'employment',
      'label'          => 'Employment',
      'contact_type_a' => 'person',
      'contact_type_b' => 'organization',
      'asymmetric'     => TRUE,
    ]);
    $this->employmentType->save();

    $this->friendshipType = RelationshipType::create([
      'id'             => 'friendship',
      'label'          => 'Friendship',
      'contact_type_a' => 'person',
      'contact_type_b' => 'person',
      'asymmetric'     => FALSE,
    ]);
    $this->friendshipType->save();
  }

  /**
   * Tests valid relationship with different contacts of correct types.
   */
  public function testValidRelationshipWithDifferentContacts(): void {
    $person = Contact::create([
      'bundle' => 'person',
      'name'   => 'John Doe',
    ]);
    $person->save();

    $organization = Contact::create([
      'bundle' => 'organization',
      'name'   => 'Acme Corp',
    ]);
    $organization->save();

    $relationship = Relationship::create([
      'bundle'   => 'employment',
      'contacts' => [
        ['target_id' => $person->id()],
        ['target_id' => $organization->id()],
      ],
    ]);

    $violations = $relationship->validate();
    $this->assertCount(0, $violations, 'Valid relationship has no violations.');
  }

  /**
   * Tests invalid relationship where both contacts are the same.
   */
  public function testInvalidRelationshipWithSameContacts(): void {
    $person = Contact::create([
      'bundle' => 'person',
      'name'   => 'John Doe',
    ]);
    $person->save();

    $relationship = Relationship::create([
      'bundle'   => 'friendship',
      'contacts' => [
        ['target_id' => $person->id()],
        ['target_id' => $person->id()],
      ],
    ]);

    $violations = $relationship->validate();
    $this->assertCount(1, $violations, 'Relationship with same contacts has a violation.');
    $this->assertEquals('Relationship Contacts must be different.', $violations[0]->getMessage());
  }

  /**
   * Tests invalid relationship where contact A has wrong type.
   */
  public function testInvalidRelationshipWithWrongContactOneType(): void {
    $person = Contact::create([
      'bundle' => 'person',
      'name'   => 'John Doe',
    ]);
    $person->save();

    $organization = Contact::create([
      'bundle' => 'organization',
      'name'   => 'Acme Corp',
    ]);
    $organization->save();

    // Create employment relationship but swap the contacts
    // (organization should be contact_b, person should be contact_a).
    $relationship = Relationship::create([
      'bundle'   => 'employment',
      'contacts' => [
        ['target_id' => $organization->id()],
        ['target_id' => $person->id()],
      ],
    ]);

    $violations = $relationship->validate();
    $this->assertGreaterThan(0, $violations->count(), 'Relationship with wrong contact A type has violations.');

    $violation_messages = [];
    foreach ($violations as $violation) {
      $violation_messages[] = (string) $violation->getMessage();
    }

    $this->assertContains('Relationship Contacts must be of type person.', $violation_messages);
  }

  /**
   * Tests invalid relationship where contact B has wrong type.
   */
  public function testInvalidRelationshipWithWrongContactTwoType(): void {
    $person1 = Contact::create([
      'bundle' => 'person',
      'name'   => 'John Doe',
    ]);
    $person1->save();

    $person2 = Contact::create([
      'bundle' => 'person',
      'name'   => 'Jane Smith',
    ]);
    $person2->save();

    // Create employment relationship but provide person for contact_b
    // (should be organization).
    $relationship = Relationship::create([
      'bundle'   => 'employment',
      'contacts' => [
        ['target_id' => $person1->id()],
        ['target_id' => $person2->id()],
      ],
    ]);

    $violations = $relationship->validate();
    $this->assertGreaterThan(0, $violations->count(), 'Relationship with wrong contact B type has violations.');

    $violation_messages = [];
    foreach ($violations as $violation) {
      $violation_messages[] = (string) $violation->getMessage();
    }

    $this->assertContains('Relationship Contacts must be of type organization.', $violation_messages);
  }

  /**
   * Tests invalid relationship where both contacts have wrong types.
   */
  public function testInvalidRelationshipWithBothContactsWrongType(): void {
    $organization1 = Contact::create([
      'bundle' => 'organization',
      'name'   => 'Acme Corp',
    ]);
    $organization1->save();

    $organization2 = Contact::create([
      'bundle' => 'organization',
      'name'   => 'TechCo Inc',
    ]);
    $organization2->save();

    // Create employment relationship but provide organizations for both
    // (should be person and organization).
    $relationship = Relationship::create([
      'bundle'   => 'employment',
      'contacts' => [
        ['target_id' => $organization1->id()],
        ['target_id' => $organization2->id()],
      ],
    ]);

    $violations = $relationship->validate();
    $this->assertGreaterThan(0, $violations->count(), 'Relationship with both contacts wrong type has violations.');

    $violation_messages = [];
    foreach ($violations as $violation) {
      $violation_messages[] = (string) $violation->getMessage();
    }

    $this->assertContains('Relationship Contacts must be of type person.', $violation_messages);
  }

  /**
   * Tests valid friendship relationship between two different persons.
   */
  public function testValidFriendshipRelationship(): void {
    $person1 = Contact::create([
      'bundle' => 'person',
      'name'   => 'John Doe',
    ]);
    $person1->save();

    $person2 = Contact::create([
      'bundle' => 'person',
      'name'   => 'Jane Smith',
    ]);
    $person2->save();

    $relationship = Relationship::create([
      'bundle'   => 'friendship',
      'contacts' => [
        ['target_id' => $person1->id()],
        ['target_id' => $person2->id()],
      ],
    ]);

    $violations = $relationship->validate();
    $this->assertCount(0, $violations, 'Valid friendship relationship has no violations.');
  }

  /**
   * Tests that NULL contacts don't cause validation errors.
   */
  public function testRelationshipWithNullContacts(): void {
    $relationship = Relationship::create([
      'bundle'   => 'employment',
      'contacts' => [],
    ]);

    // This should not throw any errors, but may have other validation issues
    // related to required fields.
    $violations = $relationship->validate();

    // We're mainly checking that it doesn't crash with NULL contacts.
    // The actual violation count may vary based on other constraints.
    $this->assertIsObject($violations);
  }

  /**
   * Tests relationship with only one contact specified.
   */
  public function testRelationshipWithOneContact(): void {
    $person = Contact::create([
      'bundle' => 'person',
      'name'   => 'John Doe',
    ]);
    $person->save();

    $relationship = Relationship::create([
      'bundle'   => 'employment',
      'contacts' => [
        ['target_id' => $person->id()],
      ],
    ]);

    // This should not throw any errors, but may have validation issues.
    $violations = $relationship->validate();

    // We're mainly checking that it doesn't crash with only one contact.
    $this->assertIsObject($violations);
  }

}
