<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Kernel;

use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\crm\Entity\Contact;
use Drupal\crm\Entity\UserContact;
use Drupal\name\Entity\NameFormat;
use Drupal\user\Entity\User;

/**
 * Tests user display name functionality.
 *
 * @group crm
 */
class UserDisplayNameTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'address',
    'comment',
    'crm',
    'inline_entity_form',
    'primary_entity_reference',
    'datetime',
    'name',
    'telephone',
  ];

  /**
   * The user data service.
   *
   * @var \Drupal\user\UserDataInterface
   */
  protected $userData;

  /**
   * The config factory service.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * Test user account.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $testUser;

  /**
   * Test contact entity.
   *
   * @var \Drupal\crm\Entity\Contact
   */
  protected $testContact;

  /**
   * Test user contact mapping.
   *
   * @var \Drupal\crm\Entity\UserContact
   */
  protected $testUserContact;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installSchema('user', ['users_data']);
    $this->installEntitySchema('comment');
    $this->installSchema('comment', ['comment_entity_statistics']);
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_contact_detail');
    $this->installEntitySchema('crm_user_contact');

    $this->installConfig(['crm', 'name']);

    $this->userData = $this->container->get('user.data');
    $this->configFactory = $this->container->get('config.factory');

    User::create([
      'name' => 'user1',
      'mail' => 'user1@example.com',
    ])->save();

    // Create a test user.
    $this->testUser = User::create([
      'name' => 'test_user',
      'mail' => 'test@example.com',
    ]);
    $this->testUser->save();

    // Create a person contact.
    $this->testContact = Contact::create([
      'bundle' => 'person',
      'name' => 'Test Contact',
      'full_name' => [
        'title' => 'Mr.',
        'given' => 'John',
        'middle' => 'Q',
        'family' => 'Doe',
        'generational' => 'Jr.',
      ],
      'preferred_name' => 'Johnny',
      'aliases' => [
        ['value' => 'JD'],
        ['value' => 'John D'],
      ],
    ]);
    $this->testContact->save();

    // Create user contact mapping.
    $this->testUserContact = UserContact::create([
      'user' => $this->testUser->id(),
      'crm_contact' => $this->testContact->id(),
    ]);
    $this->testUserContact->save();
  }

  /**
   * Tests user display name when feature is disabled.
   */
  public function testDisplayNameDisabled(): void {
    // Disable the display name override feature.
    $config = $this->configFactory->getEditable('crm.user_contact.settings');
    $config->set('display_name', FALSE);
    $config->save();

    $original_name = $this->testUser->getDisplayName();
    $name = $this->testUser->getDisplayName();

    // Name should not be altered when feature is disabled.
    $this->assertEquals($original_name, $name);
    $this->assertEquals('test_user', $name);
  }

  /**
   * Tests user display name when feature is enabled.
   */
  public function testDisplayNameEnabled(): void {
    // Enable the display name override feature.
    $config = $this->configFactory->getEditable('crm.user_contact.settings');
    $config->set('display_name', TRUE);
    $config->save();

    $name = $this->testUser->getDisplayName();

    // Name should be the contact's label when feature is enabled.
    $this->assertEquals($this->testContact->label(), $name);
  }

  /**
   * Tests user display name with no associated contact.
   */
  public function testDisplayNameNoContact(): void {
    // Enable the display name override feature.
    $config = $this->configFactory->getEditable('crm.user_contact.settings');
    $config->set('display_name', TRUE);
    $config->save();

    // Create a user without an associated contact.
    $user = User::create([
      'name' => 'no_contact_user',
      'mail' => 'nocontact@example.com',
    ]);
    $user->save();

    $name = $user->getDisplayName();

    // Name should remain unchanged when no contact is associated.
    $this->assertEquals('no_contact_user', $name);
  }

  /**
   * Tests custom name format without permission.
   */
  public function testCustomNameFormatWithoutPermission(): void {
    // Enable the display name override feature.
    $config = $this->configFactory->getEditable('crm.user_contact.settings');
    $config->set('display_name', TRUE);
    $config->save();

    // Set custom name format in user data.
    $this->userData->set('crm', $this->testUser->id(), 'name_format', 'given_family');

    $name = $this->testUser->getDisplayName();

    // Custom format should be ignored without permission.
    $this->assertEquals($this->testContact->label(), $name);
  }

  /**
   * Tests custom name format with permission.
   */
  public function testCustomNameFormatWithPermission(): void {
    // Enable the display name override feature.
    $config = $this->configFactory->getEditable('crm.user_contact.settings');
    $config->set('display_name', TRUE);
    $config->save();

    // Create a name format for testing.
    $name_format = NameFormat::create([
      'id' => 'test_format',
      'label' => 'Test Format',
      'pattern' => 't g m f',
    ]);
    $name_format->save();

    // Grant the user permission to alter display name.
    $role = $this->createRole(['crm user alter display name']);
    $this->testUser->addRole($role);
    $this->testUser->save();

    // Set custom name format in user data.
    $this->userData->set('crm', $this->testUser->id(), 'name_format', 'test_format');

    // Set the current user context.
    $this->container->get('current_user')->setAccount($this->testUser);

    $name = $this->testUser->getDisplayName();

    // Name should be formatted using the custom format.
    // The test format should produce: "Mr. John Q Doe".
    $this->assertStringContainsString('John', $name);
    $this->assertStringContainsString('Doe', $name);
  }

  /**
   * Tests name formatting with missing name components.
   */
  public function testNameFormattingWithMissingComponents(): void {
    // Enable the display name override feature.
    $config = $this->configFactory->getEditable('crm.user_contact.settings');
    $config->set('display_name', TRUE);
    $config->save();

    // Delete the previous mapping to avoid conflicts.
    $this->testUserContact->delete();

    // Create a contact with minimal name information.
    $minimal_contact = Contact::create([
      'bundle' => 'person',
      'name' => 'Minimal Contact',
      'full_name' => [
        'given' => 'Jane',
        'family' => 'Smith',
      ],
    ]);
    $minimal_contact->save();

    // Create user contact mapping.
    $minimal_user_contact = UserContact::create([
      'user' => $this->testUser->id(),
      'crm_contact' => $minimal_contact->id(),
    ]);
    $minimal_user_contact->save();

    $name = $this->testUser->getDisplayName();

    // Should fall back to contact label.
    $this->assertEquals($minimal_contact->label(), $name);
  }

  /**
   * Tests name formatting with empty full_name field.
   */
  public function testNameFormattingWithEmptyFullName(): void {
    // Enable the display name override feature.
    $config = $this->configFactory->getEditable('crm.user_contact.settings');
    $config->set('display_name', TRUE);
    $config->save();

    // Grant the user permission to alter display name.
    $role = $this->createRole(['crm user alter display name']);
    $this->testUser->addRole($role);
    $this->testUser->save();

    // Delete the previous mapping to avoid conflicts.
    $this->testUserContact->delete();

    // Create a contact with no full_name data.
    $no_name_contact = Contact::create([
      'bundle' => 'person',
      'name' => 'No Full Name Contact',
    ]);
    $no_name_contact->save();

    // Create user contact mapping.
    $no_name_user_contact = UserContact::create([
      'user' => $this->testUser->id(),
      'crm_contact' => $no_name_contact->id(),
    ]);
    $no_name_user_contact->save();

    // Set custom name format in user data.
    $this->userData->set('crm', $this->testUser->id(), 'name_format', 'given_family');

    // Set the current user context.
    $this->container->get('current_user')->setAccount($this->testUser);

    $name = $this->testUser->getDisplayName();

    // Should fall back to contact label when full_name is empty.
    $this->assertEquals($no_name_contact->label(), $name);
  }

}
