<?php

declare(strict_types=1);

namespace Drupal\Tests\crm\Unit\Plugin\Field\FieldFormatter;

use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Field\FieldDefinitionInterface;
use Drupal\Core\Field\FieldItemInterface;
use Drupal\crm\Field\AgeFieldItemList;
use Drupal\crm\Plugin\Field\FieldFormatter\AgeFormatter;
use Drupal\Tests\UnitTestCase;

/**
 * Tests the AgeFormatter.
 *
 * @group crm
 * @coversDefaultClass \Drupal\crm\Plugin\Field\FieldFormatter\AgeFormatter
 */
class AgeFormatterTest extends UnitTestCase {

  /**
   * The formatter under test.
   *
   * @var \Drupal\crm\Plugin\Field\FieldFormatter\AgeFormatter
   */
  protected $formatter;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $container = new ContainerBuilder();
    $string_translation = $this->getStringTranslationStub();
    $container->set('string_translation', $string_translation);
    \Drupal::setContainer($container);

    $configuration = [];
    $plugin_id = 'age_formatter';
    $plugin_definition = [
      'field_types' => ['integer'],
    ];
    $view_mode = '';
    $label = '';
    $field_definition = $this->createMock(FieldDefinitionInterface::class);
    $third_party_settings = [];
    $this->formatter = new AgeFormatter(
      $plugin_id,
      $plugin_definition,
      $field_definition,
      $configuration,
      $label,
      $view_mode,
      $third_party_settings,
    );
  }

  /**
   * Tests the default settings.
   *
   * @covers ::defaultSettings
   */
  public function testDefaultSettings() {
    $expected = ['granularity' => 'years'];
    $this->assertEquals($expected, AgeFormatter::defaultSettings());
  }

  /**
   * Tests the settings form.
   *
   * @covers ::settingsForm
   */
  public function testSettingsForm() {
    $form = [];
    $form_state = $this->createMock('Drupal\Core\Form\FormStateInterface');

    $settings_form = $this->formatter->settingsForm($form, $form_state);

    $this->assertArrayHasKey('granularity', $settings_form);
    $this->assertEquals('select', $settings_form['granularity']['#type']);
    $this->assertArrayHasKey('years', $settings_form['granularity']['#options']);
    $this->assertArrayHasKey('months', $settings_form['granularity']['#options']);
    $this->assertArrayHasKey('days', $settings_form['granularity']['#options']);
    $this->assertArrayHasKey('full', $settings_form['granularity']['#options']);
    $this->assertEquals('years', $settings_form['granularity']['#default_value']);
  }

  /**
   * Tests the settings summary.
   *
   * @covers ::settingsSummary
   */
  public function testSettingsSummary() {
    $summary = $this->formatter->settingsSummary();
    $this->assertIsArray($summary);
    $this->assertCount(1, $summary);
  }

  /**
   * Tests formatting with years granularity.
   *
   * @covers ::viewElements
   */
  public function testViewElementsYears() {
    // Test various day values for years granularity.
    $test_cases = [
      0 => '0 years',
      1 => '0 years',
      364 => '0 years',
      365 => '1 year',
      366 => '1 year',
      730 => '2 years',
      1095 => '3 years',
    ];

    foreach ($test_cases as $days => $expected) {
      $this->formatter->setSetting('granularity', 'years');
      $items = $this->createFieldItemList([$days]);
      $elements = $this->formatter->viewElements($items, 'en');

      $this->assertEquals($expected, $elements[0]['#markup'],
        "Failed for $days days, expected '$expected'");
    }
  }

  /**
   * Tests formatting with months granularity.
   *
   * @covers ::viewElements
   */
  public function testViewElementsMonths() {
    // Test various day values for months granularity.
    $test_cases = [
      0 => '0 months',
      29 => '0 months',
      30 => '1 month',
      31 => '1 month',
      60 => '2 months',
      365 => '12 months',
    // 12 months + 1 month (30 days)
      395 => '13 months',
    ];

    foreach ($test_cases as $days => $expected) {
      $this->formatter->setSetting('granularity', 'months');
      $items = $this->createFieldItemList([$days]);
      $elements = $this->formatter->viewElements($items, 'en');

      $this->assertEquals($expected, $elements[0]['#markup'],
        "Failed for $days days, expected '$expected'");
    }
  }

  /**
   * Tests formatting with days granularity.
   *
   * @covers ::viewElements
   */
  public function testViewElementsDays() {
    // Test various day values for days granularity.
    $test_cases = [
      0 => '0 days',
      1 => '1 day',
      2 => '2 days',
      365 => '365 days',
      1000 => '1000 days',
    ];

    foreach ($test_cases as $days => $expected) {
      $this->formatter->setSetting('granularity', 'days');
      $items = $this->createFieldItemList([$days]);
      $elements = $this->formatter->viewElements($items, 'en');

      $this->assertEquals($expected, $elements[0]['#markup'],
        "Failed for $days days, expected '$expected'");
    }
  }

  /**
   * Tests formatting with full granularity.
   *
   * @covers ::viewElements
   */
  public function testViewElementsFull() {
    // Test various day values for full granularity.
    $test_cases = [
      0 => '',
      1 => '1 day',
      30 => '1 month',
      31 => '1 month, 1 day',
      60 => '2 months',
      365 => '1 year',
      366 => '1 year, 1 day',
    // 1 year + 30 days = 1 year + 1 month
      395 => '1 year, 1 month',
    // 1 year + 31 days = 1 year + 1 month + 1 day
      396 => '1 year, 1 month, 1 day',
      730 => '2 years',
    // 2 years + 31 days
      761 => '2 years, 1 month, 1 day',
    ];

    foreach ($test_cases as $days => $expected) {
      $this->formatter->setSetting('granularity', 'full');
      $items = $this->createFieldItemList([$days]);
      $elements = $this->formatter->viewElements($items, 'en');

      $this->assertEquals($expected, $elements[0]['#markup'],
        "Failed for $days days, expected '$expected'");
    }
  }

  /**
   * Tests formatting with multiple field items.
   *
   * @covers ::viewElements
   */
  public function testViewElementsMultipleItems() {
    $this->formatter->setSetting('granularity', 'years');
    $items = $this->createFieldItemList([365, 730, 1095]);
    $elements = $this->formatter->viewElements($items, 'en');

    $this->assertCount(3, $elements);
    $this->assertEquals('1 year', $elements[0]['#markup']);
    $this->assertEquals('2 years', $elements[1]['#markup']);
    $this->assertEquals('3 years', $elements[2]['#markup']);
  }

  /**
   * Tests edge cases with invalid granularity setting.
   *
   * @covers ::viewElements
   */
  public function testViewElementsInvalidGranularity() {
    $this->formatter->setSetting('granularity', 'invalid');
    // 1 year, 1 month, 1 day
    $items = $this->createFieldItemList([396]);
    $elements = $this->formatter->viewElements($items, 'en');

    // Should default to full format.
    $this->assertEquals('1 year, 1 month, 1 day', $elements[0]['#markup']);
  }

  /**
   * Creates a mock field item list with the given values.
   *
   * @param array $values
   *   The values to create items for.
   *
   * @return \Drupal\Core\Field\FieldItemListInterface
   *   The mock field item list.
   */
  protected function createFieldItemList(array $values) {
    $items = [];
    foreach ($values as $delta => $value) {
      $item = $this->createMock(FieldItemInterface::class);
      $item->expects($this->any())
        ->method('__get')
        ->with('value')
        ->willReturn($value);
      $item->value = $value;
      $items[$delta] = $item;
    }

    $field_item_list = $this->createMock(AgeFieldItemList::class);
    $field_item_list->expects($this->any())
      ->method('getIterator')
      ->willReturn(new \ArrayIterator($items));

    return $field_item_list;
  }

}
