<?php

declare(strict_types=1);

namespace Drupal\Tests\crm_case\Functional;

use Drupal\crm_case\Entity\CrmCaseType;

/**
 * Tests the creation of CRM cases.
 *
 * @group crm_case
 */
class CrmCaseCreationTest extends CrmCaseTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'crm_case',
    'dblog',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a case type for testing.
    $case_type = CrmCaseType::create([
      'id' => 'test_case',
      'label' => 'Test Case',
    ]);
    $case_type->save();

    // Create a contact for testing.
    $contact = \Drupal::entityTypeManager()->getStorage('crm_contact')->create([
      'bundle' => 'person',
      'full_name' => ['given' => 'Test', 'family' => 'User'],
    ]);
    $contact->save();

    $web_user = $this->drupalCreateUser([
      'view crm case',
      'create crm case',
      'edit crm case',
    ]);
    $this->drupalLogin($web_user);
  }

  /**
   * Tests that a case can be created.
   */
  public function testCaseCreation(): void {
    $this->drupalGet('crm/case/add');
    $this->assertSession()->statusCodeEquals(200);

    // Create a case.
    $edit = [];
    $edit['label[0][value]'] = $this->randomMachineName(8);
    $this->drupalGet('crm/case/add/test_case');
    $this->submitForm($edit, 'Save');

    // Check that the case has been created.
    $this->assertSession()
      ->pageTextContains('New crm case ' . $edit['label[0][value]'] . ' has been created.');

    // Verify that the creation message contains a link to the case.
    $this->assertSession()
      ->elementExists('xpath', '//div[@data-drupal-messages]//a[contains(@href, "crm/case/")]');

    // Check that the case exists in the database.
    $case = $this->drupalGetCaseByLabel($edit['label[0][value]']);
    $this->assertNotEmpty($case, 'Case found in database.');

    // Verify that pages show the case information.
    $this->drupalGet('crm/case/' . $case->id());
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains($edit['label[0][value]']);
    $this->assertSession()
      ->pageTextContains($this->container->get('date.formatter')
        ->format($case->get('changed')->value));
  }

  /**
   * Tests case creation with contact assignment.
   */
  public function testCaseCreationWithContact(): void {
    // Get the test contact created in setUp().
    $contacts = \Drupal::entityTypeManager()
      ->getStorage('crm_contact')
      ->loadByProperties(['full_name.given' => 'Test']);
    $contact = reset($contacts);

    $edit = [
      'label[0][value]' => 'Case with Contact',
      'contact_id[0][target_id]' => $contact->label() . ' (' . $contact->id() . ')',
    ];

    $this->drupalGet('crm/case/add/test_case');
    $this->submitForm($edit, 'Save');

    $this->assertSession()
      ->pageTextContains('New crm case Case with Contact has been created.');

    // Verify the contact is associated.
    $case = $this->drupalGetCaseByLabel('Case with Contact');
    $this->assertEquals($contact->id(), $case->get('contact_id')->target_id);
  }

  /**
   * Tests case creation with status settings.
   */
  public function testCaseCreationWithStatus(): void {
    $edit = [
      'label[0][value]' => 'Disabled Case',
      'status[value]' => FALSE,
    ];

    $this->drupalGet('crm/case/add/test_case');
    $this->submitForm($edit, 'Save');

    $case = $this->drupalGetCaseByLabel('Disabled Case');
    $this->assertEquals(0, $case->get('status')->value, 'Case status should be disabled.');
  }

  /**
   * Tests case creation validation.
   */
  public function testCaseCreationValidation(): void {
    // Try to create a case without a label.
    $edit = [
      'label[0][value]' => '',
    ];

    $this->drupalGet('crm/case/add/test_case');
    $this->submitForm($edit, 'Save');

    // Should show validation error.
    $this->assertSession()
      ->pageTextContains('Label field is required.');

    // Verify no case was created.
    $cases = \Drupal::entityTypeManager()
      ->getStorage('crm_case')
      ->loadByProperties(['label' => '']);
    $this->assertEmpty($cases, 'No case should be created with empty label.');
  }

  /**
   * Creates cases with different authored dates.
   */
  public function testAuthoredDate(): void {
    $now = \Drupal::time()->getRequestTime();
    $admin = $this->drupalCreateUser([], NULL, TRUE);
    $this->drupalLogin($admin);

    // Create a case with the default creation date.
    $edit = [
      'label[0][value]' => $this->randomMachineName(8),
    ];
    $this->drupalGet('crm/case/add/test_case');
    $this->submitForm($edit, 'Save');
    $case = $this->drupalGetCaseByLabel($edit['label[0][value]']);
    $this->assertNotNull($case->get('created')->value);

    // Create a case with a custom creation date in the past.
    $date = $now - 86400;
    $edit = [
      'label[0][value]' => $this->randomMachineName(8),
      'created[0][value][date]' => date('Y-m-d', $date),
      'created[0][value][time]' => date('H:i:s', $date),
    ];
    $this->drupalGet('crm/case/add/test_case');
    $this->submitForm($edit, 'Save');
    $case = $this->drupalGetCaseByLabel($edit['label[0][value]']);
    $this->assertEquals($date, $case->get('created')->value);

    // Test an invalid date.
    $edit = [
      'label[0][value]' => $this->randomMachineName(8),
      'created[0][value][date]' => '2013-13-13',
      'created[0][value][time]' => '11:00:00',
    ];
    $this->drupalGet('crm/case/add/test_case');
    $this->submitForm($edit, 'Save');
    $this->assertSession()->pageTextContains('The Created date is invalid.');
    $this->assertNull($this->drupalGetCaseByLabel($edit['label[0][value]']));
  }

  /**
   * Check crm/case/add when no case types exist.
   */
  public function testCaseAddWithoutCaseTypes(): void {
    $this->drupalGet('crm/case/add');
    $this->assertSession()->statusCodeEquals(200);

    // Test /crm/case/add page without case types.
    foreach (\Drupal::entityTypeManager()->getStorage('crm_case_type')->loadMultiple() as $entity) {
      $entity->delete();
    }

    $this->drupalGet('crm/case/add');
    $this->assertSession()->statusCodeEquals(403);

    $admin_case_types = $this->drupalCreateUser([
      'administer crm',
    ]);
    $this->drupalLogin($admin_case_types);

    $this->drupalGet('crm/case/add');
    $this->assertSession()->linkByHrefExists('/admin/structure/crm_case_types/add');
  }

}
