<?php

declare(strict_types=1);

namespace Drupal\Tests\crm_case\Functional;

use Drupal\crm_case\Entity\CrmCase;
use Drupal\crm_case\Entity\CrmCaseType;

/**
 * Tests the CRM case edit form.
 *
 * @group crm_case
 */
class CrmCaseEditFormTest extends CrmCaseTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'crm_case',
    'dblog',
  ];

  /**
   * A test case entity.
   *
   * @var \Drupal\crm_case\CrmCaseInterface
   */
  protected $case;

  /**
   * A test contact entity.
   *
   * @var \Drupal\crm\CrmContactInterface
   */
  protected $contact;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a case type for testing.
    $case_type = CrmCaseType::create([
      'id' => 'edit_test',
      'label' => 'Edit Test',
    ]);
    $case_type->save();

    // Create a contact for testing.
    $this->contact = \Drupal::entityTypeManager()->getStorage('crm_contact')->create([
      'bundle' => 'person',
      'full_name' => ['given' => 'Test', 'family' => 'Contact'],
    ]);
    $this->contact->save();

    // Create a test case.
    $this->case = CrmCase::create([
      'bundle' => 'edit_test',
      'label' => 'Original Case Label',
      'contact_id' => $this->contact->id(),
      'status' => TRUE,
    ]);
    $this->case->save();

    $web_user = $this->drupalCreateUser([
      'view crm case',
      'edit crm case',
    ]);
    $this->drupalLogin($web_user);
  }

  /**
   * Tests editing a case.
   */
  public function testCaseEdit(): void {
    $this->drupalGet('crm/case/' . $this->case->id() . '/edit');
    $this->assertSession()->statusCodeEquals(200);

    // Test that the form shows the current values.
    $this->assertSession()->fieldValueEquals('label[0][value]', 'Original Case Label');
    $this->assertSession()->checkboxChecked('status[value]');

    // Edit the case.
    $edit = [
      'label[0][value]' => 'Updated Case Label',
      'status[value]' => FALSE,
    ];
    $this->submitForm($edit, 'Save');

    // Check that the case has been updated.
    $this->assertSession()
      ->pageTextContains('The crm case Updated Case Label has been updated.');

    // Verify the case was updated in the database.
    $updated_case = \Drupal::entityTypeManager()
      ->getStorage('crm_case')
      ->loadUnchanged($this->case->id());

    $this->assertEquals('Updated Case Label', $updated_case->label());
    $this->assertEquals(0, $updated_case->get('status')->value);

    // Verify we're redirected to the case page.
    $this->assertSession()->addressEquals('crm/case/' . $this->case->id());
  }

  /**
   * Tests editing case contact assignment.
   */
  public function testCaseContactEdit(): void {
    // Create another contact.
    $new_contact = \Drupal::entityTypeManager()->getStorage('crm_contact')->create([
      'bundle' => 'person',
      'full_name' => ['given' => 'New', 'family' => 'Contact'],
    ]);
    $new_contact->save();

    $this->drupalGet('crm/case/' . $this->case->id() . '/edit');

    // Change the contact assignment.
    $edit = [
      'contact_id[0][target_id]' => $new_contact->label() . ' (' . $new_contact->id() . ')',
    ];
    $this->submitForm($edit, 'Save');

    // Verify the contact was updated.
    $updated_case = \Drupal::entityTypeManager()
      ->getStorage('crm_case')
      ->loadUnchanged($this->case->id());

    $this->assertEquals($new_contact->id(), $updated_case->get('contact_id')->target_id);
  }

  /**
   * Tests case edit validation.
   */
  public function testCaseEditValidation(): void {
    $this->drupalGet('crm/case/' . $this->case->id() . '/edit');

    // Try to save with empty label.
    $edit = [
      'label[0][value]' => '',
    ];
    $this->submitForm($edit, 'Save');

    // Should show validation error.
    $this->assertSession()
      ->pageTextContains('Label field is required.');

    // Verify the case wasn't updated.
    $unchanged_case = \Drupal::entityTypeManager()
      ->getStorage('crm_case')
      ->loadUnchanged($this->case->id());

    $this->assertEquals('Original Case Label', $unchanged_case->label());
  }

  /**
   * Tests case edit with revisions.
   */
  // Public function testCaseEditWithRevisions(): void {
  //   $original_revision_id = $this->case->getRevisionId();
  // $this->drupalGet('crm/case/' . $this->case->id() . '/edit');
  // $edit = [
  //     'label[0][value]' => 'Revised Case Label',
  //     'revision[value]' => TRUE,
  //   ];
  //   $this->submitForm($edit, 'Save');
  // // Load the updated case.
  //   $updated_case = \Drupal::entityTypeManager()
  //     ->getStorage('crm_case')
  //     ->loadUnchanged($this->case->id());
  // // Should have a new revision.
  //   $this->assertNotEquals($original_revision_id, $updated_case->getRevisionId());
  //   $this->assertEquals('Revised Case Label', $updated_case->label());
  // // Verify the original revision still exists.
  //   $original_case = \Drupal::entityTypeManager()
  //     ->getStorage('crm_case')
  //     ->loadRevision($original_revision_id);
  // $this->assertEquals('Original Case Label', $original_case->label());
  // }

  /**
   * Tests case edit access permissions.
   */
  public function testCaseEditAccess(): void {
    // Test with user without edit permissions.
    $view_only_user = $this->drupalCreateUser(['view crm case']);
    $this->drupalLogin($view_only_user);

    $this->drupalGet('crm/case/' . $this->case->id() . '/edit');
    $this->assertSession()->statusCodeEquals(403);

    // Test with admin user.
    $admin_user = $this->drupalCreateUser(['administer crm']);
    $this->drupalLogin($admin_user);

    $this->drupalGet('crm/case/' . $this->case->id() . '/edit');
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests owner field changes during edit.
   */
  public function testCaseOwnerEdit(): void {
    // Create another user.
    $new_owner = $this->drupalCreateUser(['view crm case']);

    // Login as admin to edit owner field.
    $admin = $this->drupalCreateUser([], NULL, TRUE);
    $this->drupalLogin($admin);

    $this->drupalGet('crm/case/' . $this->case->id() . '/edit');

    $edit = [
      'uid[0][target_id]' => $new_owner->getAccountName() . ' (' . $new_owner->id() . ')',
    ];
    $this->submitForm($edit, 'Save');

    // Verify the owner was updated.
    $updated_case = \Drupal::entityTypeManager()
      ->getStorage('crm_case')
      ->loadUnchanged($this->case->id());

    $this->assertEquals($new_owner->id(), $updated_case->getOwnerId());
  }

}
