<?php

declare(strict_types=1);

namespace Drupal\Tests\crm_case\Functional;

use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Session\AccountInterface;
use Drupal\Tests\BrowserTestBase;
use Drupal\crm_case\CrmCaseInterface;

/**
 * Base class for CRM Case functional tests.
 */
abstract class CrmCaseTestBase extends BrowserTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = ['crm', 'crm_case', 'datetime'];

  /**
   * The default theme.
   *
   * @var string
   */
  protected $defaultTheme = 'stark';

  /**
   * The access control handler.
   *
   * @var \Drupal\Core\Entity\EntityAccessControlHandlerInterface
   */
  protected $accessHandler;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->accessHandler = \Drupal::entityTypeManager()
      ->getAccessControlHandler('crm_case');
  }

  /**
   * Asserts that case access correctly grants or denies access.
   *
   * @param array $ops
   *   An associative array of the expected case access grants for the case
   *   and account, with each key as the name of an operation (e.g. 'view',
   *   'delete') and each value a Boolean indicating whether access to that
   *   operation should be granted.
   * @param \Drupal\crm_case\CrmCaseInterface $case
   *   The case object to check.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account for which to check access.
   *
   * @internal
   */
  public function assertCaseAccess(array $ops, CrmCaseInterface $case, AccountInterface $account): void {
    foreach ($ops as $op => $result) {
      $this->assertEquals(
        $this->accessHandler->access($case, $op, $account),
        $result,
        $this->caseAccessAssertMessage($op, $result, $case->language()->getId())
      );
    }
  }

  /**
   * Asserts that case create access correctly grants or denies access.
   *
   * @param string $bundle
   *   The case bundle to check access to.
   * @param bool $result
   *   Whether access should be granted or not.
   * @param \Drupal\Core\Session\AccountInterface $account
   *   The user account for which to check access.
   * @param string|null $langcode
   *   (optional) The language code indicating which translation of the case
   *   to check. If NULL, the untranslated (fallback) access is checked.
   *
   * @internal
   */
  public function assertCaseCreateAccess(string $bundle, bool $result, AccountInterface $account, ?string $langcode = NULL): void {
    $this->assertEquals(
      $this->accessHandler->createAccess($bundle, $account, ['langcode' => $langcode]),
      $result,
      $this->caseAccessAssertMessage('create', $result, $langcode)
    );
  }

  /**
   * Constructs an assert message to display which case access was tested.
   *
   * @param string $operation
   *   The operation to check access for.
   * @param bool $result
   *   Whether access should be granted or not.
   * @param string|null $langcode
   *   (optional) The language code indicating which translation of the case
   *   to check. If NULL, the untranslated (fallback) access is checked.
   *
   * @return \Drupal\Component\Render\FormattableMarkup
   *   An assert message string which contains information in plain English
   *   about the case access permission test that was performed.
   */
  public function caseAccessAssertMessage(string $operation, bool $result, ?string $langcode = NULL): FormattableMarkup {
    return new FormattableMarkup(
      'Case access returns @result with operation %op, language code %langcode.',
      [
        '@result' => $result ? 'true' : 'false',
        '%op' => $operation,
        '%langcode' => !empty($langcode) ? $langcode : 'empty',
      ]
    );
  }

  /**
   * Returns the case by label.
   *
   * @param string $label
   *   The case label.
   *
   * @return \Drupal\crm_case\CrmCaseInterface|null
   *   The case entity or NULL if not found.
   */
  protected function drupalGetCaseByLabel(string $label): ?CrmCaseInterface {
    $cases = \Drupal::entityTypeManager()
      ->getStorage('crm_case')
      ->loadByProperties(['label' => $label]);
    return reset($cases) ?: NULL;
  }

}
