<?php

declare(strict_types=1);

namespace Drupal\Tests\crm_case\Kernel;

use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\crm_case\Entity\CrmCase;
use Drupal\crm_case\Entity\CrmCaseType;
use Drupal\user\Entity\Role;

/**
 * Tests crm case entity access control.
 *
 * @group crm_case
 */
class CrmCaseAccessTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'address',
    'name',
    'telephone',
    'crm_case',
    'crm_field',
    'comment',
    'datetime',
  ];

  /**
   * The access control handler.
   *
   * @var \Drupal\Core\Entity\EntityAccessControlHandlerInterface
   */
  protected $accessHandler;

  /**
   * A test case entity.
   *
   * @var \Drupal\crm_case\CrmCaseInterface
   */
  protected $case;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_case');
    $this->installConfig(['crm', 'crm_case', 'user']);

    $this->accessHandler = \Drupal::entityTypeManager()
      ->getAccessControlHandler('crm_case');

    // Create a case type.
    $case_type = CrmCaseType::create([
      'id' => 'access_test',
      'label' => 'Access Test',
    ]);
    $case_type->save();

    // Create a test case.
    $this->case = CrmCase::create([
      'bundle' => 'access_test',
      'label' => 'Test Access Case',
    ]);
    $this->case->save();
  }

  /**
   * Tests view access to CrmCase entities.
   */
  public function testViewAccess(): void {
    // Test anonymous user (no permissions).
    // $anonymous = $this->createUser();
    // $this->assertFalse(
    //   $this->accessHandler->access($this->case, 'view', $anonymous, TRUE)->isAllowed(),
    //   'Anonymous user should not have view access.'
    // );.
    // Test user with view permission.
    $view_user = $this->createUser(['view crm case']);
    $this->assertTrue(
      $this->accessHandler->access($this->case, 'view', $view_user, TRUE)->isAllowed(),
      'User with view permission should have view access.'
    );
  }

  /**
   * Tests update access to CrmCase entities.
   */
  public function testUpdateAccess(): void {
    // Test anonymous user (no permissions).
    // $anonymous = $this->createUser();
    // $this->assertFalse(
    //   $this->accessHandler->access($this->case, 'update', $anonymous),
    //   'Anonymous user should not have update access.'
    // );.
    // Test user with edit permission.
    $edit_user = $this->createUser(['edit crm case']);
    $this->assertTrue(
      $this->accessHandler->access($this->case, 'update', $edit_user),
      'User with edit permission should have update access.'
    );

    // Test user with administer permission.
    $admin_user = $this->createUser(['administer crm']);
    $this->assertTrue(
      $this->accessHandler->access($this->case, 'update', $admin_user),
      'User with administer permission should have update access.'
    );
  }

  /**
   * Tests delete access to CrmCase entities.
   */
  public function testDeleteAccess(): void {
    // Test anonymous user (no permissions).
    // $anonymous = $this->createUser();
    // $this->assertFalse(
    //   $this->accessHandler->access($this->case, 'delete', $anonymous),
    //   'Anonymous user should not have delete access.'
    // );.
    // Test user with delete permission.
    $delete_user = $this->createUser(['delete crm case']);
    $this->assertTrue(
      $this->accessHandler->access($this->case, 'delete', $delete_user),
      'User with delete permission should have delete access.'
    );

    // Test user with administer permission.
    $admin_user = $this->createUser(['administer crm']);
    $this->assertTrue(
      $this->accessHandler->access($this->case, 'delete', $admin_user),
      'User with administer permission should have delete access.'
    );
  }

  /**
   * Tests create access for CrmCase entities.
   */
  public function testCreateAccess(): void {
    // Test anonymous user (no permissions).
    // $anonymous = $this->createUser();
    // $this->assertFalse(
    //   $this->accessHandler->createAccess('access_test', $anonymous, [], TRUE)->isAllowed(),
    //   'Anonymous user should not have create access.'
    // );.
    // Test user with create permission.
    $create_user = $this->createUser(['create crm case']);
    $this->assertTrue(
      $this->accessHandler->createAccess('access_test', $create_user),
      'User with create permission should have create access.'
    );

    // Test user with administer permission.
    $admin_user = $this->createUser(['administer crm']);
    $this->assertTrue(
      $this->accessHandler->createAccess('access_test', $admin_user),
      'User with administer permission should have create access.'
    );
  }

  /**
   * Tests access with unknown operation.
   */
  public function testUnknownOperationAccess(): void {
    $user = $this->createUser(['view crm case']);
    $result = $this->accessHandler->access($this->case, 'unknown_operation', $user, TRUE);

    // Unknown operations should return neutral access.
    $this->assertTrue($result->isNeutral(), 'Unknown operations should return neutral access.');
  }

  /**
   * Tests access with multiple permissions.
   */
  public function testMultiplePermissionsAccess(): void {
    // Create user with multiple relevant permissions.
    $multi_user = $this->createUser([
      'view crm case',
      'edit crm case',
      'delete crm case',
    ]);

    $this->assertTrue(
      $this->accessHandler->access($this->case, 'view', $multi_user),
      'User with multiple permissions should have view access.'
    );

    $this->assertTrue(
      $this->accessHandler->access($this->case, 'update', $multi_user),
      'User with multiple permissions should have update access.'
    );

    $this->assertTrue(
      $this->accessHandler->access($this->case, 'delete', $multi_user),
      'User with multiple permissions should have delete access.'
    );
  }

  /**
   * Tests role-based permissions.
   */
  public function testRoleBasedAccess(): void {
    // Create a custom role with case permissions.
    $role = Role::create([
      'id' => 'case_manager',
      'label' => 'Case Manager',
    ]);
    $role->grantPermission('view crm case');
    $role->grantPermission('edit crm case');
    $role->save();

    // Create user and assign the role.
    $user = $this->createUser();
    $user->addRole('case_manager');
    $user->save();

    $this->assertTrue(
      $this->accessHandler->access($this->case, 'view', $user, TRUE)->isAllowed(),
      'User with role-based permissions should have view access.'
    );

    $this->assertTrue(
      $this->accessHandler->access($this->case, 'update', $user, TRUE)->isAllowed(),
      'User with role-based permissions should have update access.'
    );

    // $this->assertFalse(
    //   $this->accessHandler->access($this->case, 'delete', $user, TRUE)->isAllowed(),
    //   'User without delete permission should not have delete access.'
    // );
  }

}
