<?php

declare(strict_types=1);

namespace Drupal\Tests\crm_case\Kernel;

use Drupal\KernelTests\Core\Entity\EntityKernelTestBase;
use Drupal\crm_case\Entity\CrmCase;
use Drupal\crm_case\Entity\CrmCaseType;

/**
 * Tests crm case entity creation and base functionality.
 *
 * @group crm_case
 */
class CrmCaseCreationTest extends EntityKernelTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'address',
    'name',
    'telephone',
    'crm_case',
    'crm_field',
    'comment',
    'datetime',
  ];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $this->installEntitySchema('user');
    $this->installEntitySchema('crm_contact');
    $this->installEntitySchema('crm_case');
    $this->installConfig(['crm', 'crm_case']);
    $this->installConfig(['name']);
  }

  /**
   * Tests CrmCase entity creation.
   */
  public function testCrmCaseCreation(): void {
    // Create a case type first.
    $case_type = CrmCaseType::create([
      'id' => 'test_case',
      'label' => 'Test Case',
    ]);
    $case_type->save();

    // Create a contact to associate with the case.
    $contact = \Drupal::entityTypeManager()->getStorage('crm_contact')->create([
      'bundle' => 'person',
      'full_name' => ['given' => 'John', 'family' => 'Doe'],
    ]);
    $contact->save();

    // Create a CRM case.
    $case = CrmCase::create([
      'bundle' => 'test_case',
      'label' => 'Test Case Label',
      'contact_id' => $contact->id(),
      'status' => TRUE,
    ]);

    $this->assertInstanceOf(CrmCase::class, $case);
    $this->assertEquals('test_case', $case->bundle());
    $this->assertEquals('Test Case Label', $case->label());
    $this->assertEquals($contact->id(), $case->get('contact_id')->target_id);
    $this->assertTrue($case->get('status')->value);

    // Test that the case can be saved.
    $result = $case->save();
    $this->assertEquals(SAVED_NEW, $result);

    // Verify the case was saved with an ID.
    $this->assertNotEmpty($case->id());
    $this->assertTrue(is_numeric($case->id()));

    // Verify default owner is set to anonymous user.
    $this->assertEquals(0, $case->getOwnerId());

    // Verify timestamps are set.
    $this->assertNotEmpty($case->get('created')->value);
    $this->assertNotEmpty($case->get('changed')->value);

    // Verify revision fields.
    $this->assertNotEmpty($case->getRevisionId());
  }

  /**
   * Tests CrmCase entity field definitions.
   */
  public function testCrmCaseFieldDefinitions(): void {
    $field_definitions = \Drupal::service('entity_field.manager')
      ->getFieldDefinitions('crm_case', 'crm_case');

    // Test required base fields.
    $required_fields = [
      'id',
      'revision_id',
      'bundle',
      'uuid',
      'label',
      'status',
      'uid',
      'contact_id',
      'created',
      'changed',
    ];

    foreach ($required_fields as $field_name) {
      $this->assertArrayHasKey($field_name, $field_definitions, "Field '$field_name' should exist.");
    }

    // Test label field properties.
    $label_field = $field_definitions['label'];
    $this->assertEquals('string', $label_field->getType());
    $this->assertTrue($label_field->isRequired());
    $this->assertTrue($label_field->isRevisionable());
    $this->assertEquals(255, $label_field->getSetting('max_length'));

    // Test status field properties.
    $status_field = $field_definitions['status'];
    $this->assertEquals('boolean', $status_field->getType());
    $this->assertTrue($status_field->isRevisionable());
    // $this->assertTrue($status_field->getDefaultValue()[0]['value']);
    // Test contact_id field properties.
    $contact_field = $field_definitions['contact_id'];
    $this->assertEquals('entity_reference', $contact_field->getType());
    $this->assertEquals('crm_contact', $contact_field->getSetting('target_type'));
    $this->assertTrue($contact_field->isRevisionable());

    // Test uid field properties.
    $uid_field = $field_definitions['uid'];
    $this->assertEquals('entity_reference', $uid_field->getType());
    $this->assertEquals('user', $uid_field->getSetting('target_type'));
    $this->assertTrue($uid_field->isRevisionable());
  }

  /**
   * Tests CrmCase entity with different owners.
   */
  public function testCrmCaseOwnership(): void {
    // Create a case type.
    $case_type = CrmCaseType::create([
      'id' => 'ownership_test',
      'label' => 'Ownership Test',
    ]);
    $case_type->save();

    // Create a user.
    $user = $this->createUser();

    // Create a case with specific owner.
    $case = CrmCase::create([
      'bundle' => 'ownership_test',
      'label' => 'Owned Case',
      'uid' => $user->id(),
    ]);
    $case->save();

    $this->assertEquals($user->id(), $case->getOwnerId());
    $this->assertEquals($user->id(), $case->get('uid')->target_id);

    // Test that EntityOwnerTrait methods work.
    $this->assertEquals($user->id(), $case->getOwner()->id());
    $case->setOwner($this->createUser());
    $this->assertNotEquals($user->id(), $case->getOwnerId());
  }

  /**
   * Tests CrmCase entity revisions.
   */
  public function testCrmCaseRevisions(): void {
    // Create a case type.
    $case_type = CrmCaseType::create([
      'id' => 'revision_test',
      'label' => 'Revision Test',
    ]);
    $case_type->save();

    // Create a case.
    $case = CrmCase::create([
      'bundle' => 'revision_test',
      'label' => 'Original Label',
    ]);
    $case->save();

    $original_revision_id = $case->getRevisionId();

    // Update the case to create a new revision.
    $case->set('label', 'Updated Label');
    $case->setNewRevision(TRUE);
    $case->save();

    $new_revision_id = $case->getRevisionId();

    // Verify a new revision was created.
    $this->assertNotEquals($original_revision_id, $new_revision_id);
    $this->assertEquals('Updated Label', $case->label());

    // Load the original revision.
    $original_case = \Drupal::entityTypeManager()
      ->getStorage('crm_case')
      ->loadRevision($original_revision_id);

    $this->assertEquals('Original Label', $original_case->label());
  }

  /**
   * Tests CrmCase entity validation.
   */
  public function testCrmCaseValidation(): void {
    // Create a case type first.
    $case_type = CrmCaseType::create([
      'id' => 'validation_test',
      'label' => 'Validation Test',
    ]);
    $case_type->save();

    // Create a case without required fields to test validation.
    $case = CrmCase::create([
      'bundle' => 'validation_test',
      // Intentionally omit the 'label' field which is required.
    ]);

    $violations = $case->validate();
    $this->assertGreaterThan(0, $violations->count());

    // Check for label validation (required field).
    $label_violations = $violations->getByField('label');
    $this->assertGreaterThan(0, $label_violations->count());
  }

  /**
   * Tests CrmCase bundle validation.
   */
  public function testCrmCaseBundleValidation(): void {
    // Create a case type.
    $case_type = CrmCaseType::create([
      'id' => 'bundle_test',
      'label' => 'Bundle Test',
    ]);
    $case_type->save();

    // Create a case with valid bundle.
    $case = CrmCase::create([
      'bundle' => 'bundle_test',
      'label' => 'Valid Bundle Case',
    ]);

    $violations = $case->validate();
    $bundle_violations = $violations->getByField('bundle');
    $this->assertEquals(0, $bundle_violations->count());

    // Test that valid bundle case can be saved successfully.
    $result = $case->save();
    $this->assertEquals(SAVED_NEW, $result);

    // Test that case type must exist before creating cases.
    $case_types = \Drupal::entityTypeManager()
      ->getStorage('crm_case_type')
      ->loadMultiple();
    $this->assertArrayHasKey('bundle_test', $case_types);

    // Verify the case was created with the correct bundle.
    $this->assertEquals('bundle_test', $case->bundle());
  }

}
