<?php

declare(strict_types=1);

namespace Drupal\Tests\crm_case\Functional;

use Drupal\crm_case\Entity\CrmCase;
use Drupal\crm_case\Entity\CrmCaseType;

/**
 * Tests CRM case permissions and access control.
 *
 * @group crm_case
 */
class CrmCasePermissionTest extends CrmCaseTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'crm',
    'crm_case',
  ];

  /**
   * A test case entity.
   *
   * @var \Drupal\crm_case\CrmCaseInterface
   */
  protected $case;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a case type for testing.
    $case_type = CrmCaseType::create([
      'id' => 'permission_test',
      'label' => 'Permission Test',
    ]);
    $case_type->save();

    // Create a test case.
    $this->case = CrmCase::create([
      'bundle' => 'permission_test',
      'label' => 'Permission Test Case',
    ]);
    $this->case->save();
  }

  /**
   * Tests view permission for CRM cases.
   */
  public function testViewPermission(): void {
    // User without permission should not see the case.
    $user_no_perm = $this->drupalCreateUser();
    $this->drupalLogin($user_no_perm);

    $this->drupalGet('crm/case/' . $this->case->id());
    $this->assertSession()->statusCodeEquals(403);

    // User with view permission should see the case.
    $user_view = $this->drupalCreateUser(['view crm case']);
    $this->drupalLogin($user_view);

    $this->drupalGet('crm/case/' . $this->case->id());
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Permission Test Case');
  }

  /**
   * Tests create permission for CRM cases.
   */
  public function testCreatePermission(): void {
    // User without permission should not access creation forms.
    $user_no_perm = $this->drupalCreateUser();
    $this->drupalLogin($user_no_perm);

    $this->drupalGet('crm/case/add');
    $this->assertSession()->statusCodeEquals(403);

    $this->drupalGet('crm/case/add/permission_test');
    $this->assertSession()->statusCodeEquals(403);

    // User with create permission should access creation forms.
    $user_create = $this->drupalCreateUser(['create crm case']);
    $this->drupalLogin($user_create);

    $this->drupalGet('crm/case/add');
    $this->assertSession()->statusCodeEquals(200);

    $this->drupalGet('crm/case/add/permission_test');
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests edit permission for CRM cases.
   */
  public function testEditPermission(): void {
    // User without permission should not access edit forms.
    $user_no_perm = $this->drupalCreateUser(['view crm case']);
    $this->drupalLogin($user_no_perm);

    $this->drupalGet('crm/case/' . $this->case->id() . '/edit');
    $this->assertSession()->statusCodeEquals(403);

    // User with edit permission should access edit forms.
    $user_edit = $this->drupalCreateUser(['edit crm case']);
    $this->drupalLogin($user_edit);

    $this->drupalGet('crm/case/' . $this->case->id() . '/edit');
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests delete permission for CRM cases.
   */
  public function testDeletePermission(): void {
    // User without permission should not access delete forms.
    $user_no_perm = $this->drupalCreateUser(['view crm case']);
    $this->drupalLogin($user_no_perm);

    $this->drupalGet('crm/case/' . $this->case->id() . '/delete');
    $this->assertSession()->statusCodeEquals(403);

    // User with delete permission should access delete forms.
    $user_delete = $this->drupalCreateUser(['delete crm case']);
    $this->drupalLogin($user_delete);

    $this->drupalGet('crm/case/' . $this->case->id() . '/delete');
    $this->assertSession()->statusCodeEquals(200);
    $this->assertSession()->pageTextContains('Are you sure you want to delete');
  }

  /**
   * Tests admin permission access.
   */
  public function testAdminPermission(): void {
    $admin_user = $this->drupalCreateUser(['administer crm']);
    $this->drupalLogin($admin_user);

    // Admin should have all access.
    $this->drupalGet('crm/case/' . $this->case->id());
    $this->assertSession()->statusCodeEquals(200);

    $this->drupalGet('crm/case/add');
    $this->assertSession()->statusCodeEquals(200);

    $this->drupalGet('crm/case/' . $this->case->id() . '/edit');
    $this->assertSession()->statusCodeEquals(200);

    $this->drupalGet('crm/case/' . $this->case->id() . '/delete');
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests collection page access.
   */
  public function testCollectionAccess(): void {
    // User without admin permission should not access collection.
    $user_no_admin = $this->drupalCreateUser(['view crm case']);
    $this->drupalLogin($user_no_admin);

    $this->drupalGet('admin/content/crm/case');
    $this->assertSession()->statusCodeEquals(403);

    // Admin user should access collection.
    $admin_user = $this->drupalCreateUser(['administer crm']);
    $this->drupalLogin($admin_user);

    $this->drupalGet('admin/content/crm/case');
    $this->assertSession()->statusCodeEquals(200);
  }

  /**
   * Tests multiple permissions work together.
   */
  public function testMultiplePermissions(): void {
    $multi_user = $this->drupalCreateUser([
      'view crm case',
      'create crm case',
      'edit crm case',
    ]);
    $this->drupalLogin($multi_user);

    // Should have view access.
    $this->drupalGet('crm/case/' . $this->case->id());
    $this->assertSession()->statusCodeEquals(200);

    // Should have create access.
    $this->drupalGet('crm/case/add/permission_test');
    $this->assertSession()->statusCodeEquals(200);

    // Should have edit access.
    $this->drupalGet('crm/case/' . $this->case->id() . '/edit');
    $this->assertSession()->statusCodeEquals(200);

    // Should NOT have delete access.
    $this->drupalGet('crm/case/' . $this->case->id() . '/delete');
    $this->assertSession()->statusCodeEquals(403);
  }

}
