<?php

declare(strict_types=1);

namespace Drupal\crowdsec\Form;

use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Logger\RfcLogLevel;
use Drupal\crowdsec\ScenarioPluginManager;
use Drupal\crowdsec\Signals;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Configure crowdsec settings for this site.
 */
class Settings extends ConfigFormBase {

  /**
   * The signals service.
   *
   * @var \Drupal\crowdsec\Signals
   */
  protected Signals $signals;

  /**
   * The scenario plugin manager.
   *
   * @var \Drupal\crowdsec\ScenarioPluginManager
   */
  protected ScenarioPluginManager $scenarioPluginManager;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): Settings {
    $instance = parent::create($container);
    $instance->signals = $container->get('crowdsec.signals');
    $instance->scenarioPluginManager = $container->get('plugin.manager.crowdsec_scenario');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'crowdsec_settings';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['crowdsec.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $config = $this->config('crowdsec.settings');

    $form['log_level'] = [
      '#type' => 'select',
      '#title' => $this->t('Log level'),
      '#options' => RfcLogLevel::getLevels(),
      '#default_value' => $config->get('log_level'),
      '#weight' => -20,
      '#description' => $this->t('Select the log level to be used for logging module messages. Higher levels capture more detailed logs, while lower levels capture only essential messages.'),
    ];

    $form['env'] = [
      '#type' => 'select',
      '#title' => $this->t('Environment'),
      '#options' => [
        'dev' => $this->t('Development'),
        'prod' => $this->t('Production'),
      ],
      '#default_value' => $config->get('env'),
      '#required' => TRUE,
      '#description' => $this->t('Select the environment type. "Development" enables more detailed debugging and verbose logging for testing purposes, while "Production" optimizes performance and restricts logging to only critical issues. Switching the environment will reset the site’s ID and result in the loss of historical data.'),
    ];
    $form['warning'] = [
      '#type' => 'item',
      '#markup' => '<p><strong>' . $this->t('Attention: if you change the environment, CrowdSec will have to assign a new ID to this Drupal site and all legacy data will be lost.') . '</strong></p>',
      '#states' => [
        'invisible' => [
          'select[name="env"]' => ['value' => $config->get('env')],
        ],
      ],
    ];

    $form['api_timeout'] = [
      '#type' => 'number',
      '#title' => $this->t('API Timeout'),
      '#default_value' => $config->get('api_timeout'),
      '#required' => TRUE,
      '#min' => -1,
      '#max' => 999,
      '#description' => $this->t('Specify the timeout in seconds for the API request to CrowdSec. A negative value will disable the timeout. If the timeout occurs, the request to CrowdSec fails, and the module logs a critical error. Cached data is used as a fallback to continue processing, which may affect the accuracy of remediation decisions. If no cache is available, the default behavior will bypass remediation.'),
    ];

    $form['cti_api_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('CTI API Key'),
      '#default_value' => $config->get('cti_api_key'),
      '#description' => $this->t('Provide your CTI API Key. This is optional and only required if you want to receive smoke data about IP addresses. You can <a href=":api_url" target="_blank">obtain your key</a> when you <a href=":account_url" target="_blank">create a free account</a>.', [
        ':api_url' => 'https://app.crowdsec.net/settings/cti-api-keys',
        ':account_url' => 'https://app.crowdsec.net',
      ]),
    ];

    foreach ($this->scenarioPluginManager->getDefinitions() as $definition) {
      $id = $definition['id'];
      $plugin = $this->scenarioPluginManager->createInstance($id);
      $form[$id] = [
        '#type' => 'details',
        '#title' => $plugin->label(),
        '#description' => $plugin->getPluginDefinition()['description'],
        '#open' => TRUE,
        '#tree' => TRUE,
      ];
      $form[$id]['enable'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable'),
        '#default_value' => $plugin->getSetting('enable'),
      ];
      $form[$id]['details'] = [
        '#type' => 'container',
        '#states' => [
          'visible' => [
            ':input[name="' . $id . '[enable]"]' => ['checked' => TRUE],
          ],
        ],
      ];
      if ($plugin->getPluginDefinition()['buffer']) {
        $form[$id]['details']['leak_speed'] = [
          '#type' => 'number',
          '#title' => $this->t('Time period'),
          '#description' => $this->t('Time period in seconds in which the number of bad requests (see threshold below) need to be recognized to get the IP banned.'),
          '#default_value' => $plugin->getSetting('leak_speed'),
          '#min' => 1,
        ];
        $form[$id]['details']['bucket_capacity'] = [
          '#type' => 'number',
          '#title' => $this->t('Threshold'),
          '#description' => $this->t('Number of bad requests in any given time period needed to get the IP banned.'),
          '#default_value' => $plugin->getSetting('bucket_capacity'),
          '#min' => 1,
        ];
      }
      $form[$id]['details']['ban_duration'] = [
        '#type' => 'number',
        '#title' => $this->t('Duration'),
        '#description' => $this->t('Time period in seconds for how long the IP should remain banned.'),
        '#default_value' => $plugin->getSetting('ban_duration'),
        '#min' => 1,
      ];
    }

    $form['signal_scenarios'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Signal scenarios'),
      '#options' => $this->signals->signalScenarios(),
      '#default_value' => $config->get('signal_scenarios'),
      '#description' => $this->t('Choose the scenarios that will signal IPs to be blocked by CrowdSec.'),
    ];

    $form['scenarios'] = [
      '#type' => 'checkboxes',
      '#title' => $this->t('Subscribe scenarios'),
      '#options' => $this->signals->scenarios(),
      '#default_value' => $config->get('scenarios'),
      '#description' => $this->t('Select which attack scenarios you want your Drupal site to subscribe to. These scenarios will trigger action against suspicious IPs.'),
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);
    if (empty($this->cleanupMap($form_state->getValue('scenarios')))) {
      $form_state->setErrorByName('scenarios', $this->t('At least one scenario has to be enabled.'));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    $config = $this->config('crowdsec.settings');
    $config
      ->set('log_level', $form_state->getValue('log_level'))
      ->set('env', $form_state->getValue('env'))
      ->set('api_timeout', $form_state->getValue('api_timeout'))
      ->set('cti_api_key', $form_state->getValue('cti_api_key'))
      ->set('signal_scenarios', $this->cleanupMap($form_state->getValue('signal_scenarios')))
      ->set('scenarios', $this->cleanupMap($form_state->getValue('scenarios')));
    foreach ($this->scenarioPluginManager->getDefinitions() as $definition) {
      $id = $definition['id'];
      $plugin = $this->scenarioPluginManager->createInstance($id);
      $config
        ->set($plugin->getSettingKey('enable'), $form_state->getValue([
          $id,
          'enable',
        ]))
        ->set($plugin->getSettingKey('ban_duration'), $form_state->getValue([
          $id,
          'details',
          'ban_duration',
        ]));
      if ($definition['buffer']) {
        $config
          ->set($plugin->getSettingKey('leak_speed'), $form_state->getValue([
            $id,
            'details',
            'leak_speed',
          ]))
          ->set($plugin->getSettingKey('bucket_capacity'), $form_state->getValue([
            $id,
            'details',
            'bucket_capacity',
          ]));
      }
    }
    $config->save();
    parent::submitForm($form, $form_state);
  }

  /**
   * Transforms a map into an array of selected elements.
   *
   * @param array $map
   *   The map.
   *
   * @return array
   *   The resulting list.
   */
  private function cleanupMap(array $map): array {
    $result = [];
    foreach ($map as $item) {
      if ($item) {
        $result[] = $item;
      }
    }
    return $result;
  }

}
