<?php

declare(strict_types=1);

namespace Drupal\crowdsec\Hook;

use Drupal\Core\Form\FormState;
use Drupal\Core\Hook\Attribute\Hook;
use Drupal\crowdsec\Buffer;
use Drupal\crowdsec\Client;
use Drupal\crowdsec\ScenarioPluginManager;

/**
 * Implements hooks for the CrowdSec module.
 */
class CrowdsecHooks {

  /**
   * Constructs a new CrowdsecHooks object.
   */
  public function __construct(
    protected Buffer $buffer,
    protected Client $client,
    protected readonly ScenarioPluginManager $scenarioPluginManager,
  ) {}

  /**
   * Implements hook_help().
   */
  #[Hook('help')]
  public function help(string $route_name): string {
    $output = '';
    if ($route_name === 'help.page.crowdsec') {
      $output .= '<h3>' . t('About') . '</h3>';
      $output .= '<p>' . t('Integrates your Drupal site with CrowdSec to keep suspicious users and cybercriminals away. For more information, see the <a href=":url">online documentation for the CrowdSec module</a>.', [
        ':url' => 'https://www.drupal.org/docs/contributed-modules/crowdsec',
      ]) . '</p>';
      $output .= '<p>' . t('You can configure this module <a href=":config">here</a>.', [
        ':config' => '/admin/config/services/crowdsec',
      ]) . '</p>';
    }
    return $output;
  }

  /**
   * Implements hook_cron().
   */
  #[Hook('cron')]
  public function cron(): void {
    $this->buffer->push();
    $this->client->refresh();
  }

  /**
   * Implements hook_form_FORM_ID_alter().
   */
  #[Hook('form_ban_ip_form_alter')]
  public function alterBanIpForm(array &$form): void {
    $form['#submit'][] = [self::class, 'submitBanIpForm'];
  }

  /**
   * Submit handler for the ban_ip_form.
   *
   * @param array $form
   *   The form.
   * @param \Drupal\Core\Form\FormState $form_state
   *   The form state.
   */
  public static function submitBanIpForm(array $form, FormState $form_state): void {
    $ip = $form_state->getValue('ip');
    if ($ip) {
      ScenarioPluginManager::getPlugin('ban')->addSignal($ip, 200);
    }
  }

}
