<?php

declare(strict_types=1);

namespace Drupal\crowdsec;

use CrowdSec\Common\Constants;
use Drupal\Component\Render\FormattableMarkup;
use Drupal\Core\Logger\LoggerChannelInterface;
use Drupal\crowdsec\Event\CrowdSecEvents;
use Drupal\crowdsec\Event\IpBlocked;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\HttpKernelInterface;

/**
 * Provides a HTTP middleware to implement CrowdSec based IP blocking.
 */
class Middleware implements HttpKernelInterface {

  /**
   * Flags if the middleware is active, can be disabled with ::disable().
   *
   * @var bool
   */
  protected static bool $active = TRUE;

  /**
   * Constructs a CrowdSec Middleware object.
   */
  public function __construct(
    protected HttpKernelInterface $httpKernel,
    protected Client $client,
    protected LoggerChannelInterface $logger,
    protected EventDispatcherInterface $eventDispatcher,
  ) {}

  /**
   * Turns off this middleware for the current request.
   */
  public static function disable(): void {
    self::$active = FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function handle(Request $request, $type = HttpKernelInterface::MAIN_REQUEST, $catch = TRUE): Response {
    if (self::$active && $this->client->isConfigured()) {
      $ip = $request->getClientIp();
      if ($ip !== NULL && $this->client->verifyIp($ip) === Constants::REMEDIATION_BAN) {
        $this->logger->notice('Blocked @ip, details see <a href="@url">dashboard</a>', [
          'ip' => $ip,
          '@ip' => $ip,
          '@url' => Client::CROWDSEC_URL_CTI . $ip,
        ]);
        $this->eventDispatcher->dispatch(new IpBlocked($ip), CrowdSecEvents::IP_BLOCKED);
        return new Response((new FormattableMarkup('@ip has been blocked', ['@ip' => $ip]))->__toString(), 403);
      }
    }
    return $this->httpKernel->handle($request, $type, $catch);
  }

}
